-- PLANUNGS-TABELLEN

-- Auftragsbegleitkarte
-- >  https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen
CREATE TABLE abk (
  --ab_trs_id             VARCHAR(100),                                                 -- Auflösungsid ABK Erstellung
  --ab_plan_id            VARCHAR(100),                                                 -- Darstellung in Plantafel?????
  ab_ix                 serial PRIMARY KEY,                                             -- ABK-Index, Hauptidentifikator
  ab_parentabk          integer REFERENCES abk ON UPDATE CASCADE ON DELETE CASCADE INITIALLY DEFERRED CONSTRAINT xtt29861 CHECK( ab_parentabk IS DISTINCT FROM ab_ix ), -- ParentABK, übergeordnete ABK zu dieser, Default NULL wenn kein übergeordnete Struktur vorhanden
  ab_mainabk            integer REFERENCES abk ON UPDATE CASCADE ON DELETE CASCADE INITIALLY DEFERRED, -- MainABK, Haupt-ABK in der obersten Ebene. Gibt es keine übergeordnete Ebene, dann steht hier der eigene ABK-Index.
  --
  ab_ap_nr              varchar(40),                                                    -- Arbeitspaket-Nr., Zeichenlänge wie Artikel, kann auch auch Projektnummer oder ähnliches enthalten
  ab_ap_bem             varchar(100),                                                   -- Bemerkung/Text Arbeitspaket
  ab_ap_txt             text,                                                           -- Langtext Arbeitspaket
  ab_ap_txt_rtf         text,
  --
  ab_tablename          varchar(100),                                                   -- zugehörige Tabelle
  ab_dbrid              varchar(100),                                                   -- Record-ID bzgl. zugehöriger Tabelle
  ab_keyvalue           varchar(50),                                                    -- P-Key oder UNIQUE-Key der verlinkten Tabelle (Auftrags-Nr., Bestell-Nr., Artikel-Nr., Kurzname, QAB-Nr. ...)
  --
  ab_plan_ag_id         integer REFERENCES auftg ON DELETE CASCADE,                     -- Vorab-Einplanung: Verkaufs-ag_id. https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Plantafel; https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen
  ab_displaycaption     varchar(100),                                                   -- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen
  --
  ab_an_nr              varchar(50) REFERENCES anl ON UPDATE CASCADE,                   -- redundantes Feld, da ab_ix in anl. Für Suche erforderlich, da F2-Fenster mit Rekursiv-Suche zu langsam
  ab_done               boolean NOT NULL DEFAULT false,                                    -- erledigt: Alle AG beendet bzw, manuell angeklickt (Teilprojekt). Schliesst Materiallisten und AG
  ab_done_dat           timestamp(0),
  ab_storno             boolean NOT NULL DEFAULT false,                                    -- stonriert

  -- Defaults für Freigaben kommen aus Trigger abk__b_20_i und werden insb. anhand Zustand der ABK gesteuert.
  ab_doeinkauf          boolean NOT NULL,                                                  -- Materialliste: Freigabe Einkauf
  ab_dolager            boolean NOT NULL,                                                  -- Materialliste: Freigabe Lager
  --
  ab_pos                integer,                                                        -- Reihenfolge im Ablauf
  ab_ld_id              integer CONSTRAINT xtt1913 UNIQUE CONSTRAINT xtt2200 REFERENCES ldsdok,
  --ab_dat              DATE DEFAULT current_date,                                      -- Erstelldatum
  ab_at                 date,                                                           -- Anfangstermin über alle AG
  ab_et                 date,                                                           -- Endetermin über alle AG
  --
  ab_askix              integer REFERENCES opl,                                         -- ASK-Index
  ab_st_uf1             NUMERIC DEFAULT 1,                                              -- Fertigungsmenge (höher bei geplantem Ausschuss)
  ab_st_uf1_soll        NUMERIC(12,4),                                                  -- Sollmenge (Ziel-Fertigungsmenge bei geplantem Ausschuss)
  -------------------------------
  /*Spalten zur Nachkalkulation*/
  -------------------------------
  ab_nk_date            date,                                                           -- Datum Nachkalulation erstellt
  ab_nk_st_uf1          numeric NOT NULL DEFAULT 0,                                     -- Fertigungsmenge zurückgemeldet Nachkalkulation
  ab_nk_et              numeric(16,4),                                                  -- Gesamtkosten pro Einzelteil
  ab_nk_mat             numeric(16,4),                                                  -- Materialkosten Vorprodukte gesamt
  ab_nk_auss            numeric(12,4),                                                  -- Ausschuss

  ab_nk_matr_et         numeric(16,4),                                                  -- Rohmaterialkosten (IC3) Einzelteil                            #12214
  ab_nk_matr_bg         numeric(16,4),                                                  -- Rohmaterialkosten (IC3) Baugruppe (alles ab dieser Ebene)     #12214

  ab_nk_mgk             numeric(5,2) NOT NULL DEFAULT 0,
  ab_nk_agk             numeric(5,2) NOT NULL DEFAULT 0,
  ab_nk_fgk             numeric(5,2) NOT NULL DEFAULT 0,
  ab_nk_rgk             numeric(5,2) NOT NULL DEFAULT 0,
  ab_nk_change          boolean NOT NULL DEFAULT FALSE,                                 -- Gibt an, ob Änderungen an AG oder Mat stattfanden
  ab_matchange          boolean DEFAULT FALSE,                                          -- Materialänderung vorhanden
  ab_txt                text,                                                           -- Bemerkung z.B. für BDE-Stempelzeiten etc.
  ab_nk_txt             text,                                                           -- Bemerkung Nachkalkulation
  ---------------
  /*Statusflags*/
  ---------------
  ab_rckmeld            boolean NOT NULL DEFAULT FALSE,                                    -- Rückmeldungen vorhanden
  ab_buch               boolean NOT NULL DEFAULT FALSE,                                    -- ABK wurde verbucht
  ab_print              boolean DEFAULT FALSE,                                             -- gedruckt
  ab_stat               varchar(300),                                                   -- Status (EAV) für sonstiges, z.B. Set-ABK (http://redmine.prodat-sql.de/issues/7237) und QS-Aufgaben
  ab_op_stat            varchar(3) REFERENCES oplstat,                                  -- Teilestatus
  ab_erstfert           boolean NOT NULL DEFAULT FALSE,                                    -- Erstfertigung
  ab_inplantaf          boolean NOT NULL DEFAULT FALSE,                                    -- in Plantafel

  ab_sperr              boolean NOT NULL DEFAULT FALSE,                                 -- Sperrstatus #9197
  ab_lagkomm_prio       integer,                                                        -- Priorisierung von Kommissionieraufträge #11727

  --- #19302
  ab_allg_n1            numeric (12,4),
  ab_allg_v1            varchar(75),

  -- System (tables__generate_missing_fields)  dbrid                 varchar(32) NOT NULL DEFAULT nextval('db_id_seq'),
  insert_date           date,
  insert_by             varchar(32),
  modified_by           varchar(32),
  modified_date         timestamp(0)
);

SELECT setval('abk_ab_ix_seq', 10000);

-- Indizes
  CREATE INDEX abk_ab_ix_text ON abk (CAST(ab_ix AS text) varchar_pattern_ops);

  CREATE INDEX abk_ab_parentabk ON abk (ab_parentabk) WHERE ab_parentabk IS NOT NULL;
  CREATE INDEX abk_ab_parentabk_text ON abk (CAST(ab_parentabk AS text) varchar_pattern_ops) WHERE CAST(ab_parentabk AS text) IS NOT NULL;
  CREATE INDEX abk_ab_mainabk ON abk (ab_mainabk);
  CREATE INDEX abk_ab_mainabk_text ON abk (CAST(ab_mainabk AS text) varchar_pattern_ops);

  CREATE INDEX ab_plan_ag_id ON abk(ab_plan_ag_id) WHERE ab_plan_ag_id IS NOT NULL;
  CREATE INDEX abk_ld_id ON abk(ab_ld_id) WHERE ab_ld_id IS NOT NULL;

  CREATE INDEX abk_buch ON abk(ab_ix) WHERE NOT ab_buch;
  CREATE INDEX abk_ab_done ON abk (ab_ix) WHERE NOT ab_done;

  CREATE INDEX abk_inplantaf ON abk (ab_ix) WHERE ab_inplantaf;
  CREATE INDEX abk_inplantaf_not ON abk (ab_ix) WHERE NOT ab_inplantaf;

  CREATE INDEX abk_tablenamedbrid ON abk (ab_tablename, ab_dbrid) WHERE ab_tablename IS NOT NULL;
  CREATE INDEX abk_keyvalue ON abk (ab_keyvalue varchar_pattern_ops) WHERE ab_tablename IS NOT NULL;

  CREATE INDEX abk_an_nr ON abk (ab_an_nr varchar_pattern_ops) WHERE ab_an_nr IS NOT NULL;

-- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
CREATE TRIGGER abk_set_modified
  BEFORE INSERT OR UPDATE
  ON abk
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();
--

CREATE TRIGGER abk__a_i__create_autoparams
  AFTER INSERT
  ON abk
  FOR EACH ROW
  EXECUTE PROCEDURE trecnoparam.createautoparams();
--

--KeyWordSearch
CREATE OR REPLACE FUNCTION abk__a_iu_keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
   PERFORM TSystem.kws_create_keywords(new.*);
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_iu_keywordsearch
   AFTER INSERT OR UPDATE
   OF ab_ix, ab_ap_nr, ab_an_nr, ab_keyvalue, ab_tablename, ab_dbrid -- tablename/dbrid können indirekt im trigger keyvalue setzen
   ON abk
   FOR EACH ROW
   EXECUTE PROCEDURE abk__a_iu_keywordsearch();

-- #21113 Absicherung gegen doppelt vergebene ABK-IDs
CREATE OR REPLACE FUNCTION abk__b_01_i__ab_ix() RETURNS TRIGGER AS $$
  BEGIN

    -- Wenn ABK-Index bereits vergeben wurde, dann solange die nächste Nummer ziehen bis eine freie Nummer gefunden wurde
    WHILE EXISTS( SELECT 1 FROM abk WHERE ab_ix = new.ab_ix ) LOOP
      new.ab_ix := nextval( 'abk_ab_ix_seq' );
    END LOOP;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  DROP TRIGGER IF EXISTS abk__b_01_i__ab_ix ON public.abk;
  CREATE TRIGGER abk__b_01_i__ab_ix
    BEFORE INSERT
    ON abk
    FOR EACH ROW
    WHEN ( TSystem.execution_flag__isset( _flagname => 'upsert_for_abk' ) IS false )
    EXECUTE PROCEDURE abk__b_01_i__ab_ix();
--


--https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
CREATE OR REPLACE FUNCTION abk__b_50_iu__abapnr() RETURNS TRIGGER AS $$
  BEGIN
    IF tg_op='UPDATE' THEN
        IF new.ab_ap_nr IS DISTINCT FROM old.ab_ap_nr AND (new.ab_ap_nr IS DISTINCT FROM (SELECT op_n FROM opl WHERE op_ix=new.ab_askix)) THEN --
            new.ab_askix:=NULL;
        END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_50_iu__abapnr
    BEFORE INSERT OR UPDATE OF ab_ap_nr
    ON abk
    FOR EACH ROW
    WHEN (new.ab_ap_nr IS NOT NULL)
    EXECUTE PROCEDURE abk__b_50_iu__abapnr();
--

--
CREATE OR REPLACE FUNCTION abk__b_50_iu() RETURNS TRIGGER AS $$
  BEGIN
    IF new.ab_at<current_date-(365*100) THEN --datum liegt 100 Jahre zurück => Schwachsinn (eigentlich programmfehler da nicht durch Nutzer eingegeben)
        RAISE EXCEPTION 'abk__b_iu: invalid date: %', new.ab_at;
    END IF;

    IF (new.ab_st_uf1_soll  = 0) THEN
        new.ab_st_uf1_soll:=NULL;
    END IF;

    IF new.ab_askix = -1 THEN --veralteter Quark, erstmal stehenlassen. zB Kopieren ABK Struktur > wiki DS: 2018-04-10
        new.ab_askix:=NULL;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_50_iu
    BEFORE INSERT OR UPDATE
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__b_50_iu();
--

CREATE OR REPLACE FUNCTION abk__b_90_iu() RETURNS TRIGGER AS $$
  BEGIN
    -- im allgemeinen Trigger, falls in irgendeinem Trigger zB da Menge < Menge geliefert das umgesetzt wird und es kein OF gibt!
    IF new.ab_done IS true THEN
      new.ab_done_dat := coalesce(new.ab_done_dat, now());
    ELSE
      new.ab_done_dat := null;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_90_iu
    BEFORE INSERT OR UPDATE
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__b_90_iu();

--https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen
CREATE OR REPLACE FUNCTION abk__b_11_iu__tablenamedbrid() RETURNS TRIGGER AS $$
  DECLARE
      _shema varchar;
      _str   varchar;
      _qabnr integer;
  BEGIN

      IF new.ab_ld_id IS NOT null THEN

          -- ABK hat Bezug zu einem Fertigungsauftrag, der mit einem QAB verknüpft wurde. Z.Bsp. wenn QS-Fertigung manuell angelegt wurde ...
          SELECT ld_q_nr
            INTO _qabnr
            FROM ldsdok
           WHERE ld_id = new.ab_ld_id
             AND ld_code = 'I'
             AND ld_auftg ILIKE '%QS'
          ;

          -- ... dann erzwingen wir, dass die ABK mit dem QAB Verküpft wird.
          IF _qabnr IS NOT null THEN

             SELECT 'qab', qab.dbrid, q_nr
               INTO new.ab_tablename, new.ab_dbrid, new.ab_keyvalue
               FROM qab
              WHERE q_nr = _qabnr;

          END IF;

      END IF;

      IF new.ab_tablename = 'beleg_k__artchange' THEN
        new.ab_stat = TSystem.ENUM_SetValue(new.ab_stat,'AM');  -- Änderungsmanagement / Change Management
      END IF;

      -- Das ist eine Bearbeitungsstruktur
      SELECT rck_schema, rck_column
        INTO _shema, _str
        FROM recnocommentkategorie
       WHERE rck_id = new.ab_tablename;


      IF _str IS NOT null THEN

          EXECUTE (
            'SELECT ' || _str
            || ' FROM ' || coalesce( _shema || '.', '') || new.ab_tablename
            || ' WHERE dbrid = ' || quote_literal( new.ab_dbrid )
          )
          INTO new.ab_keyvalue;

      END IF;

      -- Displaycaption; Projektbezug (ab_an_nr) SOWIE Artikel (ab_ap_nr) setzen!
      SELECT
          annr,
          coalesce( new.ab_ap_nr, aknr ),
          ident,
          coalesce( keyvalue, new.ab_keyvalue )
      INTO
          new.ab_an_nr,
          new.ab_ap_nr,
          new.ab_displaycaption,
          new.ab_keyvalue
      FROM tableident_allfields( new.ab_tablename, new.ab_dbrid );
      --

      IF new.ab_ap_nr IS null THEN

          -- an vielen Stellen wird so bei Projekten zB die Projektnummer angezeigt
          new.ab_ap_nr := new.ab_keyvalue;
      END IF;


      IF new.ab_displaycaption IS null THEN
          new.ab_displaycaption := new.ab_keyvalue;
      END IF;
      --
      RETURN new;

  END $$ LANGUAGE plpgsql;


 CREATE TRIGGER abk__b_11_iu__tablenamedbrid
  BEFORE INSERT OR UPDATE
  OF ab_tablename, ab_dbrid, ab_parentabk
  ON abk
  FOR EACH ROW
  WHEN (new.ab_tablename IS NOT NULL AND new.ab_dbrid IS NOT NULL)
  EXECUTE PROCEDURE abk__b_11_iu__tablenamedbrid();
--
CREATE OR REPLACE FUNCTION abk__b_10_iu__parentabk() RETURNS TRIGGER AS $$
 DECLARE
  parentabkchange boolean = false;
 BEGIN
  --
   IF new.ab_parentABK IS NOT NULL THEN

      --da wir im insert sind, ist der aktuelle datensatz für die funktion noch nicht sichtbar, daher parent angeben, falls da
      new.ab_mainabk := tplanterm.abk_main_abk( coalesce( new.ab_parentabk, new.ab_ix ) );

      IF ( TSystem.ENUM_GetValue( new.ab_stat,'SPL') AND new.ab_parentabk IS NOT NULL ) THEN
          new.ab_displaycaption := lang_text(16653) || ' ' || new.ab_parentabk;
      END IF;

   ELSE
      -- kein Parent, keine mainABK!
      new.ab_mainabk := new.ab_ix;
   END IF;

   --
   IF tg_op = 'UPDATE' THEN
      parentabkchange := new.ab_parentabk IS DISTINCT FROM old.ab_parentabk;
   END IF;
   --


   -- das gehört zu einem projekt, dann die zugehörige dbrid/tablename auch noch hinzu, falls noch nicht vorhanden
   IF    (new.ab_parentabk IS NOT NULL AND new.ab_tablename IS NULL )
      OR ( parentabkchange )
   THEN
       SELECT ab_tablename, ab_dbrid
       INTO new.ab_tablename, new.ab_dbrid
       FROM abk WHERE ab_ix = new.ab_mainabk;
   END IF;
   --

   RETURN new;

 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER abk__b_10_iu__parentabk
  BEFORE INSERT OR UPDATE
  OF ab_parentabk
  ON abk
  FOR EACH ROW
  EXECUTE PROCEDURE abk__b_10_iu__parentabk();

CREATE OR REPLACE FUNCTION abk__a_10_u__main_parentabk() RETURNS TRIGGER AS $$ --Hinweis: ag_parentABK usw. müssen vor TRIGGER abk__a_iud() gesetzt sein
 BEGIN
   PERFORM execution_code__disable( _flagname => 'auftg' );
   PERFORM disablemodified();
   --
   IF new.ab_mainABK IS DISTINCT FROM old.ab_mainabk THEN --kompletter Zweig verzogen
      UPDATE abk SET ab_mainABK = abkm.ab_ix, ab_tablename = abkm.ab_tablename, ab_dbrid = abkm.ab_dbrid
                 FROM abk abkm
                 WHERE abkm.ab_ix = new.ab_mainABK
                   AND abk.ab_mainabk = old.ab_mainABK
                   AND abk.ab_ix <> new.ab_ix
                   AND abk.ab_ix IN (SELECT * FROM tplanterm.get_all_child_abk(new.ab_ix));
      --
      UPDATE auftg SET ag_mainABK = new.ab_mainABK WHERE ag_mainABK = old.ab_mainABK;
   END IF;
   --falls wir die ABK im Baum schieben, müssen wir gleich das zugehörige Auftragselement mit auf den neuen Parent setzen, das ist derzeit doppelt
   --Problemfall: Auftrag hat als Ursprung eine andere ABK. Die Feritung wurde frei aufgelegt und verknüpft.
   --Durch verändern der ABK selbst, wird nun auch das Material angefasst. Da wird aber frei eingekauft sind und keine ab_parentabk haben, fliegt uns das um die Ohren
   IF new.ab_parentabk IS DISTINCT FROM old.ab_parentabk THEN--ABK wurde tatsächlich verschoben
      UPDATE auftg
         SET ag_parentabk = new.ab_parentabk,
             ag_mainabk   = new.ab_mainabk
       WHERE ag_ownabk = new.ab_ix
         AND (   ag_parentabk IS DISTINCT FROM new.ab_parentabk
              OR ag_mainabk   IS DISTINCT FROM new.ab_mainabk
             )
             ;
   END IF;
   --
   PERFORM enablemodified();
   PERFORM execution_code__enable( _flagname => 'auftg' );
  --
  RETURN new;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER abk__a_10_u__main_parentabk
  AFTER UPDATE
  OF ab_mainABK, ab_parentabk
  ON abk
  FOR EACH ROW
  EXECUTE PROCEDURE abk__a_10_u__main_parentabk();
--

--
CREATE OR REPLACE FUNCTION abk__b_20_i() RETURNS TRIGGER AS $$
  BEGIN

    -- Defaults initial per Settings
      -- Anwendereingaben initial akzeptieren.
    new.ab_doeinkauf := coalesce( new.ab_doeinkauf, TSystem.Settings__GetBool( 'abk_ab_doeinkauf_autofreigabe' ) );
    new.ab_dolager   := coalesce( new.ab_dolager,   TSystem.Settings__GetBool( 'abk_ab_dolager_autofreigabe' )   );

    IF      new.ab_ld_id IS NOT NULL
        AND new.ab_tablename IS NULL
    THEN
          SELECT coalesce( new.ab_ap_nr, ld_aknr ), ld_auftg, ld_an_nr,
                 -- Bei Projektstruktur bereits gesetzt
                 coalesce( new.ab_displaycaption , twawi.ldsdok_GenConcatNr( ldsdok ) )
            INTO new.ab_ap_nr, new.ab_keyvalue, new.ab_an_nr,
                 new.ab_displaycaption
            FROM ldsdok
                 -- das ist die artikelnummer aktuell --ACHTUNG ld_abk ist noch nicht gesetzt!
           WHERE ld_id = new.ab_ld_id;

        --ACHTUNG: BEI ÄNDERUNG BEACHTE TWawi.LdsDokI__create__from__auftgi >> DORT DAS GLEICHE
    END IF;

    IF new.ab_plan_ag_id IS NOT NULL THEN

          SELECT ag_nr, ag_an_nr,
                 -- PlanABK mit STV: aknr wird aus Oberfläche eingesteuert
                 coalesce(new.ab_ap_nr, ag_aknr),
                 twawi.auftg_GenConcatNr(auftg)
            INTO
                 new.ab_keyvalue, new.ab_an_nr,
                 new.ab_ap_nr,
                 new.ab_displaycaption
            FROM auftg
                 -- das ist die artikelnummer aktuell --ACHTUNG ld_abk ist noch nicht gesetzt!
           WHERE ag_id = new.ab_plan_ag_id;

    END IF;
    --
    IF new.ab_ap_nr IS NOT NULL THEN

          IF
              NOT EXISTS(
                SELECT true
                  FROM ldsdok
                 WHERE ld_code = 'I'
                   AND ld_aknr = new.ab_ap_nr
                   AND ld_abk <> new.ab_ix
                   AND ld_abk IS NOT NULL
              )
          THEN

              -- erstfertigung, noch keine ABK für diesen Artikel da gewesen
              new.ab_erstfert := true;
        END IF;
    END IF;

    --- Aus Einkaufsmodulen heraus die ABK immer freigeben. (Beistellung)
    --- http://redmine.prodat-sql.de/issues/3773
    IF new.ab_tablename = 'ldsdok' THEN

       new.ab_doeinkauf := true;
       new.ab_dolager   := TSystem.Settings__GetBool('abk_ab_dolager_autofreigabe');
    END IF;

    --Set-Artikel, immer freiegegeben
    -- QAB Autofreigabe
    -- siehe auch #8950
    IF
          TSystem.ENUM_GetValue( new.ab_stat, 'PS' )
       OR TSystem.ENUM_GetValue( new.ab_stat, 'STR' )
       OR (
               new.ab_tablename = 'qab'
           AND TSystem.Settings__GetBool( 'abk_ab_dolager__ab_doeinkauf__qab_autofreigabe' )
          )
    THEN
        -- Set-ABK, dann sind die Teile automatisch, wie auch bei der ABK freigegeben!
       new.ab_doeinkauf := true;
       new.ab_dolager   := true;
    END IF;

    IF new.ab_parentabk = 0 THEN

       -- wegen AsInteger im Delphi
       new.ab_parentabk := null;
    END IF;

    IF new.ab_askix = 0 THEN

       -- wegen AsInteger im Delphi
       new.ab_askix := null;
    END IF;

    RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_20_i
    BEFORE INSERT
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__b_20_i();
--

--
CREATE OR REPLACE FUNCTION abk__b_50_u() RETURNS TRIGGER AS $$
  BEGIN

    IF TSystem.current_user_in_syncro_dblink() THEN
        RETURN new;
    END IF;
    --ABK/Arbeitspaket schliessen
    IF new.ab_storno THEN
        new.ab_done:=TRUE;
    END IF;
    --
    IF new.ab_parentabk=0 THEN
        new.ab_parentabk:=NULL;--wegen AsInteger im Delphi
    END IF;
    -- Freigabe Einkauf setzen falls Setting aktiviert
    IF TSystem.Settings__GetBool('abk_ab_doeinkauf_autofreigabe_bei_Plantafel') AND new.ab_inplantaf THEN
        new.ab_doeinkauf := true;
    END IF;
    --
    IF new.ab_buch THEN
        new.ab_done:=TRUE;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_50_u
    BEFORE UPDATE
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__b_50_u();
--

--
CREATE OR REPLACE FUNCTION abk__b_50_d() RETURNS TRIGGER AS $$
  BEGIN
     DELETE FROM nkz WHERE nz_ab_ix=old.ab_ix;
     RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__b_50_d
    BEFORE DELETE
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__b_50_d();
--

--https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
--https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen
CREATE OR REPLACE FUNCTION abk__a_10_i() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    --UPDATE auftg SET ag_post2='E', ag_post4=new.ab_ix WHERE ag_id=(SELECT ld_ag_id FROM ldsdok WHERE ld_abk=new.ab_ix) AND ag_post2='N' AND ag_post4 IS NULL;--wir tragen in die Auftg den Status nach
    UPDATE ldsdok SET ld_abk = new.ab_ix WHERE ld_id = new.ab_ld_id AND ld_abk IS DISTINCT FROM new.ab_ix;
    UPDATE auftg  SET ag_ownabk = new.ab_ix FROM ldsdok WHERE ld_abk = new.ab_ix AND ag_astat = 'I' AND ag_id = ld_ag_id AND ag_ownabk IS DISTINCT FROM new.ab_ix;
    --Prognosen schliessen
    --UPDATE bestanfpos SET bap_pr_done=TRUE, bap_done=TRUE FROM ldsdok WHERE NOT bap_done AND bap_pr_ld_id=ld_id AND ld_abk=new.ab_ix;
    --
    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_10_i
    AFTER INSERT
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__a_10_i();
--

--Achtung - bei Änderungen Hints in Oberfläche "ABK Bearbeiten" anpassen.
--http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Fertigung
CREATE OR REPLACE FUNCTION abk__a_60_u__AGbeenden() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN ((new.ab_done AND NOT old.ab_done) OR (new.ab_storno <> old.ab_storno))
    -- Stornierung wird gesetzt.
    IF new.ab_storno AND NOT old.ab_storno THEN
        -- Stornieren der ABK storniert auch die Materiallistenpositionen [Ticket 4607] (Bisher: --Materialliste bleibt offen auch wenn Arbeitspaket beendet, wird beim Verbuchen geschlossen)
        UPDATE auftg SET ag_done = true, ag_storno = true WHERE NOT ag_done AND ag_parentabk = new.ab_ix;
        -- interne Bestellung stornieren
        UPDATE ldsdok SET ld_done = true, ld_storno = true WHERE NOT ld_done AND ld_abk = new.ab_ix;
    END IF;

    -- Stornierung wird aufgehoben
    IF old.ab_storno AND NOT new.ab_storno THEN
        -- interne Bestellung: Stornierung auch aufheben. Erledigt-Status nur rückgängig, wenn noch keine Lieferungen auf Position, vgl. #6467.
        UPDATE ldsdok SET ld_storno = false, ld_done = CASE WHEN ld_stkl <= 0 THEN false ELSE ld_done END WHERE ld_storno AND ld_abk = new.ab_ix;
    END IF;

    -- Fälle die nur beim Erledigt- bzw. Storniert-Setzen (ab_done durch abk__b_u) ausgeführt werden sollen, aber nicht: beim Ent-Stornieren bzw. Nicht-Erledigt-Setzen.
    IF new.ab_done AND NOT old.ab_done THEN
        -- Alle AG beenden
        UPDATE ab2 SET a2_ende = true WHERE a2_ab_ix = new.ab_ix AND NOT a2_ende;
        -- Unter-ABK abschließen, wenn Über-ABK zu Ende und Unter-ABK keine AG
        UPDATE abk SET ab_storno = new.ab_storno, ab_done = new.ab_done WHERE ab_parentabk = new.ab_ix AND NOT EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix = ab_ix) AND NOT EXISTS(SELECT true FROM auftg WHERE ag_parentabk=ab_ix);
        -- Über-ABK abschließen, wenn alle Unterabk zuende und Über-ABK keine AG und keine Materialliste

        -- Auswärtsbearbeitungen schließen // http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Ausw%C3%A4rtsbearbeitung
        UPDATE ldsdok SET ld_done = true
        WHERE ld_a2_id IN (SELECT a2_id FROM abk JOIN ab2 ON a2_ab_ix = ab_ix AND a2_ausw WHERE ab_ix = new.ab_ix)
          AND ld_auftg LIKE 'AW.%' AND ld_code = 'E'
          AND NOT ld_done;

        --Wir prüfen, ob die übergeordnete ABK noch untergeordnete, offene ABK hat
        IF NOT EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix = new.ab_parentabk) THEN
            IF NOT EXISTS(SELECT true FROM ab2 WHERE NOT a2_ende AND a2_ab_ix <> new.ab_ix AND a2_ab_ix <> new.ab_parentabk AND a2_ab_ix IN (SELECT * FROM tplanterm.get_all_child_abk(new.ab_parentabk))) THEN
                UPDATE abk SET ab_done = true, ab_storno = new.ab_storno WHERE NOT ab_done AND ab_ix = new.ab_parentabk;
            END IF;
        END IF;

        -- ABK-Lagerabgänge löschen
        DELETE FROM tlager.lagerabgang__abk__temp WHERE labk_abix = new.ab_ix;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_60_u__AGbeenden
    AFTER UPDATE
    OF ab_done, ab_storno -- Wenn Storno, dann auch ab_done (s. abk__b_u) und damit auch alle Arbeitsgänge schließen.
    ON abk
    FOR EACH ROW
    WHEN ((new.ab_done AND NOT old.ab_done) OR (new.ab_storno <> old.ab_storno))
    EXECUTE PROCEDURE abk__a_60_u__AGbeenden();
--

--
CREATE OR REPLACE FUNCTION abk__a_55_u__ab_buch() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    PERFORM execution_flag__aquire( _flagname => 'inTerminierung' );  --flag setzen damit der AB2 trigger nicht versucht die termine in die abk zu schreiben
    UPDATE ab2 SET a2_ende=TRUE, a2_interm=FALSE WHERE a2_ab_ix=new.ab_ix AND (a2_interm OR NOT a2_ende OR NOT a2_buch);
    PERFORM execution_flag__release( _flagname => 'inTerminierung' ); --flag setzen damit der AB2 trigger nicht versucht die termine in die abk zu schreiben
    --Materialliste spätestens beim verbuchen auf beendet setzen
    --Beachte Hints in Oberflächen
    --http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Fertigung
    UPDATE auftg SET ag_done=true, ag_storno=new.ab_storno WHERE NOT ag_done AND ag_parentabk=new.ab_ix;
    PERFORM enablemodified();
    RETURN new;
  END$$LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_55_u__ab_buch
    AFTER UPDATE
    OF ab_buch
    ON abk
    FOR EACH ROW
    WHEN (new.ab_buch)
    EXECUTE PROCEDURE abk__a_55_u__ab_buch();
--

--
CREATE OR REPLACE FUNCTION abk__a_55_u__ab_doeinkauf__ab_dolager() RETURNS trigger AS $$
  BEGIN
      -- Vererben des Status Freigabe Material/Lager von Kopf an Childs
      UPDATE abk
         SET ab_doeinkauf = new.ab_doeinkauf
       WHERE ab_ix IN ( SELECT * FROM tplanterm.get_all_child_abk(new.ab_ix) )
         AND ab_doeinkauf IS DISTINCT FROM new.ab_doeinkauf;

      UPDATE abk
         SET ab_dolager = new.ab_dolager
       WHERE ab_ix IN ( SELECT * FROM tplanterm.get_all_child_abk(new.ab_ix) )
         AND ab_dolager IS DISTINCT FROM new.ab_dolager;

      RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_55_u__ab_doeinkauf__ab_dolager
    AFTER UPDATE
    OF ab_doeinkauf, ab_dolager
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__a_55_u__ab_doeinkauf__ab_dolager();
--

CREATE OR REPLACE FUNCTION abk__a_80_u__ab_stat() RETURNS TRIGGER AS $$
    DECLARE _ti_type scheduling.resource_timeline_blocktype;
    BEGIN
        IF TSystem.ENUM_ContainsValue(new.ab_stat, 'BUFFER') THEN
           _ti_type := 'task.buffer';
        ELSE
           _ti_type := 'task';
        END IF;

        UPDATE scheduling.resource_timeline SET ti_type = _ti_type WHERE ti_a2_id IN (SELECT a2_id FROM ab2 WHERE a2_ab_ix = new.ab_ix /*AND NOT a2_ende*/) AND ti_type <> _ti_type;

        RETURN new;
    END $$ LANGUAGE plpgsql;
    --
    DROP TRIGGER IF EXISTS abk__a_80_u__ab_stat ON abk;
    CREATE TRIGGER abk__a_80_u__ab_stat
      AFTER UPDATE OF ab_stat
      ON abk
      FOR EACH ROW
      EXECUTE PROCEDURE abk__a_80_u__ab_stat();

--
CREATE OR REPLACE FUNCTION abk__a_iud() RETURNS TRIGGER AS $$ -- Hinweis: Zugriff auf ag_parenabk, daher muss abk__b_10_iu__parentabk vorher laufen!
  DECLARE dobedarf BOOL;
          doterm   BOOL;
          rows     INTEGER;
  BEGIN
    IF TG_OP IN ('INSERT', 'UPDATE') THEN
        PERFORM execution_code__disable( _flagname => 'auftg' );
        IF TG_OP = 'UPDATE' THEN
            -- Damit wenigstens in der Bedarfsübersicht und beim Lagerzugang die richtigen Zahlen erscheinen!
            UPDATE ldsdok SET
              ld_stk = new.ab_st_uf1, ld_stk_soll = new.ab_st_uf1_soll
            WHERE ld_abk = new.ab_ix
              AND (ld_stk IS DISTINCT FROM new.ab_st_uf1 OR ld_stk_soll IS DISTINCT FROM new.ab_st_uf1_soll);

            IF new.ab_st_uf1 IS DISTINCT FROM old.ab_st_uf1 THEN
                UPDATE ab2 SET a2_ta = NULL WHERE a2_ab_ix = new.ab_ix; -- Änderung Menge führt zu neuen AG-Zeiten.
            END IF;

            -- Projektnummer an Materialliste weitergeben. Achtung, kein OF-Trigger, da abk__b_iu__tablenamedbrid
            IF NOT Equals(new.ab_an_nr, old.ab_an_nr) THEN
                UPDATE auftg SET ag_an_nr = new.ab_an_nr WHERE ag_parentabk = new.ab_ix AND NOT Equals(ag_an_nr, new.ab_an_nr);
            END IF;
        END IF;

        -- MontageTermin
        IF new.ab_tablename = 'anl' THEN
            IF new.ab_ap_nr = TSystem.Settings__Get('PJMTERMAP') THEN --Wenn ArbeitspaketNr=Setting aus Einstellungen für MontageArbeitspaket
                UPDATE anl SET an_idat = new.ab_at WHERE dbrid = new.ab_dbrid AND an_idat IS DISTINCT FROM new.ab_at;
            END IF;
        END IF;

        -- Bedarfsberechnung, wenn wir Freigabe Einkauf ändern
        IF tg_op = 'INSERT' THEN
            dobedarf := new.ab_doeinkauf;
            doterm   := new.ab_at IS NOT NULL;
        ELSE -- dobedarf, doterm
            dobedarf := new.ab_doeinkauf <> old.ab_doeinkauf;
            doterm   :=     new.ab_at IS DISTINCT FROM old.ab_at
                         OR new.ab_et IS DISTINCT FROM old.ab_et;
        END IF;

        IF doterm THEN
            -- Bedarfs-Termine (ag_kdatum) werden in Materialliste (auftg I) gesetzt.
              -- Termin von Unterbauteilen einer anderen Fertigung (E, M) ist Ende der Fertigung des Unterbauteils.

              -- 2023-10-11: das eigene Produktionsdatum setzt NIE das Bedarfsdatum in der Stückliste selbst um. Das ergibt sich aus den Arbeitsgängen und führt zu Fehlmaterial!
              -- PERFORM tabk.abk__auftgi_kdatum__refresh_by__ag_ownabk(new.ab_ix);

              -- eigene Rohmaterialien und Normteile der Fertigung (R, N) per ABK-Materialtermin ggf. anhand planrelevanten KS, siehe #12665
              PERFORM tabk.abk__auftgi_kldatum__refresh_by__ag_parent_abk(new.ab_ix); -- ab_auftgi_kdatum wird vorher per FUNCTION tabk.abk__ab_auftgi_kdatum__calc gesetzt, siehe TRIGGER abk__b_60_iu__termini.
            --

            -- Termin in PA übertragen (ab_et nach ld_terml).
            PERFORM tabk.abk__ldsdok__term_set(new.ab_ix);

            -- Termine von Set-Artikeln (#7237): Unter-ABK ohne Bestellbezug, ähnlich Projekt, allerdings ohne AG
            UPDATE abk
               SET ab_at = new.ab_at, ab_et = new.ab_at
             WHERE ab_parentabk = new.ab_ix
               AND ab_ld_id IS NULL
               AND NOT EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix = ab_ix);
        END IF;

        IF dobedarf THEN -- Position ist bedarfswirksam
            PERFORM tartikel.bedarf__make_bedarf(ag_aknr) FROM auftg WHERE NOT ag_done AND ag_parentabk = new.ab_ix;
        END IF;

        PERFORM execution_code__enable( _flagname => 'auftg' );

        RETURN new;
    ELSE -- DELETE
        IF old.ab_buch THEN   RETURN old; END IF; -- bereits verbucht, dann raus.

        UPDATE auftg SET ag_ownabk = NULL WHERE ag_ownabk = old.ab_ix AND ag_astat = 'E'; -- ABK <> Auftragsbezug (Lieferauftrag) entfernen; #6478
        -- Bestellungen/Produktionsaufträge
        DELETE FROM ldsdok WHERE ld_abk = old.ab_ix AND ld_interncreate AND ld_stkl = 0;  -- ohne Bewegung: einfach löschen
        UPDATE ldsdok SET ld_abk = NULL, ld_stat = ENUM_DelValue(ld_stat, old.ab_stat) WHERE ld_abk = old.ab_ix AND NOT ld_interncreate; -- Vom Nutzer eingegebene Bestellungen stehen wieder zur ABK Erstellung an.
        -- Teilbewegte (?was soll das bedeuten)
        UPDATE ldsdok SET ld_abk = NULL, ld_storno = true, ld_txt = 'DELETE ABK:' || old.ab_ix || ':AP-Nr:' || COALESCE(old.ab_ap_nr, 'ab_ap_nr NULL') WHERE ld_abk = old.ab_ix AND ld_interncreate;
        -- Bedarfe/Interne Aufträge
        UPDATE ldsdok SET ld_ag_id = NULL WHERE ld_ag_id IN (SELECT ag_id FROM auftg WHERE ag_stkl = 0 /*AND ag_stkb=0 Menge eingekauft existiert immer, wenn folgeABK!*/ AND ag_parentabk = old.ab_ix AND NOT ag_done AND ag_astat = 'I'); -- wenn aus Auftrag wieder Bestellung.
        DELETE FROM auftg WHERE ag_stkl = 0 AND ag_stkb = 0 AND ag_parentabk = old.ab_ix AND ag_astat = 'I'; -- Bisher unberührte Bedarfspositionen entfernen.
        -- Falls wir selbst eine ABK aus einem Teilebedarf sind, diese Verknüpfung entfernen.
        UPDATE auftg SET ag_ownabk = NULL, ag_txt = 'DELETE ownABK:' || old.ab_ix WHERE ag_ownabk = old.ab_ix AND ag_astat = 'I'; -- Bedarfsverursacher bleibt bestehen, hat aber keine Produktion/Erfüller mehr.
        -- Wir stornieren die Unterbauteile dieser ABK! Dies sind teilbewegte, unberührte wurden oben gelöscht.
        UPDATE auftg SET ag_parentabk = NULL, ag_storno = true, ag_txt = 'DELETE ABK:' || old.ab_ix || ':AP-Nr:' || COALESCE(old.ab_ap_nr, 'ab_ap_nr NULL') WHERE ag_parentabk = old.ab_ix AND NOT ag_done AND ag_astat = 'I';

        RETURN old;
    END IF;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON abk
    FOR EACH ROW
    EXECUTE PROCEDURE abk__a_iud();
--

-- Laufende Fertigung von Werkzeug-/ Vorrichtungen wird mit auftragsbegleitenden Artikeln entspr. laufender ABK verlinkt. #8423
-- vgl. ab2ba__b_iu__abk
CREATE OR REPLACE FUNCTION abk__a_80_iu__ab2ba() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (NOT new.ab_done AND new.ab_ap_nr IS NOT NULL AND new.ab_tablename = 'anl') -- ABK läuft, Werkzeug-/ Vorrichtungsartikel ist angg. und aus Projekt für Vorrichtungsbau.
    IF EXISTS(SELECT true FROM ab2ba WHERE a2ba_ak_nr = new.ab_ap_nr AND a2ba_ab_ix IS NULL) THEN -- Performance (EXISTS ohne JOINs auf Index)
        UPDATE ab2ba SET
          a2ba_ab_ix = new.ab_ix    -- aktuelle Fertigung wird mit verwendeten auftragsbegleitenden Artikel verlinkt.
        FROM ab2
          JOIN abk ON ab_ix = a2_ab_ix
        WHERE a2_id = a2ba_a2_id
          AND NOT a2_ende           -- AG der ABK, wo Werkzeug/Vorrichtung verwendet wird, ist nicht beendet.
          AND NOT ab_done           -- Entspr. ABK ist ebenso nicht beendet.
          AND a2ba_ab_ix IS NULL    -- noch keine ABK angg.
          AND a2ba_ak_nr = new.ab_ap_nr; -- Auftragsbegleitender Artikel ist Werkzeug-/ Vorrichtungsartikel der aktuellen ABK.
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER abk__a_80_iu__ab2ba
    AFTER INSERT OR UPDATE
    OF ab_ap_nr
    ON abk
    FOR EACH ROW
    WHEN (NOT new.ab_done AND new.ab_ap_nr IS NOT NULL AND new.ab_tablename = 'anl') -- ABK läuft, Werkzeug-/ Vorrichtungsartikel ist angg. und aus Projekt für Vorrichtungsbau.
    EXECUTE PROCEDURE abk__a_80_iu__ab2ba();
--

-- Beachte CONSTRAINT TRIGGER sind immer AFTER
-- hier: AFTER INSERT mit DEFERRABLE INITIALLY DEFERRED (am Ende der Transaktion) FOR EACH ROW (für jede Zeile ausführen)
CREATE OR REPLACE FUNCTION abk__c_10_i_id__auftgi__ag_a2_id__by__stv_op2__create() RETURNS TRIGGER AS $$
  BEGIN
    -- Erzeugt Verknüpfung von Materialpositionen und Arbeitsgängen der ABK
    -- am Ende der Transaktion der ABK-Erstellung aus der Oberfläche.


    -- Ausführung des Triggers zur Aktualisierung des Materialtermins bei initialer Erstellung nicht notwendig.
    PERFORM TSystem.Settings__Set( 'disable__auftg__a_u__ag_a2_id', true );

    -- Verknüpfungen erzeugen
    PERFORM tabk.abk__auftgi__ag_a2_id__by__stv_op2__create( new.ab_ix );

    -- Ausführung des Triggers zur Aktualisierung des Materialtermins wieder aktivieren.
    PERFORM TSystem.Settings__Set( 'disable__auftg__a_u__ag_a2_id', false );


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE CONSTRAINT TRIGGER abk__c_10_i_id__auftgi__ag_a2_id__by__stv_op2__create
    AFTER INSERT
    ON abk
    -- am Ende der Transaktion
    DEFERRABLE INITIALLY DEFERRED
    FOR EACH ROW
    EXECUTE PROCEDURE abk__c_10_i_id__auftgi__ag_a2_id__by__stv_op2__create();
--

--Demontage oder Umbau oder Upgrade
CREATE OR REPLACE FUNCTION abk__a_u__demontage() RETURNS TRIGGER AS $$
    BEGIN
      --Kopfposition der Bestellung schliessen, wenn nötig
      PERFORM TQS.ldsdok__demontage_head_done__set(new.ab_ld_id);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE OR REPLACE FUNCTION TQS.abk__demontage__is(_abk abk) RETURNS boolean AS $$
    DECLARE
        _stat text;
    BEGIN
        _stat := _abk.ab_stat;
        IF _stat IS NULL THEN
          RETURN false;
        ELSE
          RETURN (   TSystem.ENUM_GetValue(_stat, 'DM')
                  OR TSystem.ENUM_GetValue(_stat, 'UB')
                  OR TSystem.ENUM_GetValue(_stat, 'UG')
                 );
        END IF;
    END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION TQS.abk__demontage__is(_abix integer) RETURNS boolean
    AS $$
        SELECT TQS.abk__demontage__is(abk)
          FROM abk
         WHERE ab_ix = _abix;
    $$ LANGUAGE sql STABLE;


CREATE TRIGGER abk__a_u__demontage -- abk__a_90_u__ab_done__demontage
    AFTER UPDATE
    OF ab_done
    ON abk
    FOR EACH ROW
    WHEN (    new.ab_done
          AND TQS.abk__demontage__is(new)
        )
    EXECUTE PROCEDURE abk__a_u__demontage();
--

-- Arbeitsgänge zu einer ABK
CREATE TABLE ab2 (
  a2_id                     serial PRIMARY KEY,
  a2_ab_ix                  integer NOT null REFERENCES abk ON UPDATE CASCADE ON DELETE CASCADE,

  a2_dlz                    integer,                                -- auswertige bearbeitungszeit in tagen
  a2_at                     timestamp without time zone,            -- Anfangstermin
  a2_et                     timestamp without time zone,            -- Endetermin
  a2_ks                     varchar(9) NOT null REFERENCES ksv ON UPDATE CASCADE,   --Kostenstelle
  a2_ksap                   varchar(50),                            -- Kostenstelle/Arbeitsplatz
  a2_interm                 boolean NOT null DEFAULT false,         -- Terminiert/Eingelastet
  a2_sperr                  boolean NOT null DEFAULT false,

  a2_v_ll_dbusename         varchar(20),                            -- verantwortlicher Mitarbeiter
  a2_n                      integer NOT null CONSTRAINT chk_a2_n_positive CHECK (a2_n > 0), --Arbeitsgangnummer
  a2_o2_id                  integer REFERENCES op2 ON DELETE SET null, --direkte op2_id zB wegen umnummerieren

  --a2_parent_a2_id   integer REFERENCES ab2 ON UPDATE CASCADE ON DELETE CASCADE,
  a2_ncnr                   varchar(20),                            -- NC-Programm-Nr.
  --a2_ll_minr                integer REFERENCES llv,
    a2_prio                   integer, -- TODO KLÄRUNG, DROP

  a2_aknr                   varchar(40) REFERENCES art ON UPDATE CASCADE, -- Arbeitspakete auf Artikelebene
  a2_ausw                   boolean NOT null DEFAULT false,         -- Auswärtsarbeitsgang
  a2_awpreis                numeric,                                -- Preis Auswärtsbearbeitung
  a2_preis_table            varchar(40),
  a2_preis_dbrid            varchar(32),
  a2_auswpruef              boolean NOT null DEFAULT false,            -- Für Auswaertsarbeitsgang wurde der Preis vom Einkauf geprüft/Freigegeben
  a2_adkrz                  varchar(21) REFERENCES adk ON UPDATE CASCADE, --Vorschlag Auswärtsbearbeiter
  a2_awtx                   varchar(100),                           --Vorschlag Auswärtskategorie

  a2_tv                     numeric DEFAULT 0,

  -- Zeiteinheitein (1=Stunden ; 2=Minuten ; 3=Sekunden ; 4=Tage ( entspricht 8 stunden , siehe tabelle zeinh und deren standardwerte ) )
  a2_zeinh_tr               integer NOT null REFERENCES zeinh DEFAULT 1, -- Rüstzeit
  a2_zeinh_tx               integer NOT null REFERENCES zeinh DEFAULT 2, -- Hauptzeit

  a2_tr                     numeric(12,4)DEFAULT 0,                 -- Rüstzeit
  a2_tr_sek                 numeric,
  a2_th                     numeric(12,4) DEFAULT 0,                -- Hauptzeit
  a2_th_sek                 numeric,
  a2_tm                     numeric(12,4) DEFAULT 0,                -- Mitarbeiterzeit
  a2_tm_sek                 numeric,
  a2_tn                     numeric(12,4) DEFAULT 0,                -- Nebenzeit
  a2_tn_sek                 numeric,
  a2_lgz                    numeric(12,4) DEFAULT 0,                -- Liegezeit
  a2_lgz_sek                numeric,
  a2_ul                     numeric(12,4),                          -- Überlappung (0..100%)

  a2_subject                varchar(100),                           -- Erste Zeile des AG-Text oder manuell eingetragen, siehe Trigger ab2__b_iu
  a2_txt                    text,                                   -- Langtext
  a2_txt_rtf                text,
  a2_txt_intern             text,

  a2_ende                   boolean NOT null DEFAULT false,         -- Beendet
  a2_ende_dat               timestamp(0),
  a2_rckmeld                boolean NOT null DEFAULT false,         -- Rückmeldungen (Zeitmeldungen) vorhanden
  a2_buch                   boolean NOT null DEFAULT false,         -- Arbeitsgang wurde verbucht

  a2_ta                     numeric(12,6),                          -- Gesamtvorgangszeit _immer_ in stunden
    -- DROP? ... auch o2_....
    a2_maschauto_ta           numeric(8,4),                           --Maximale Mannloslaufzeit des Arbeitsgangs
    -- TODO DROP a2_group_ta               numeric(12,6),                          --diese Zeit dieses arbeitsgangs ist in einem anderen Arbeitsgang untergebracht
  a2_time_stemp             numeric(12,6) NOT null DEFAULT 0,        --Zeit gemeldet total
  a2_time_stemp__ruest      numeric(12,6),                          -- Zeit gemeldet (gestempelt) NUR Rüsten
  a2_time_stemp__ruest_overlap  numeric(12,6),                      -- Zeit gemeldet: Rüstzeit ÜBER der geplanten Rüstzeit (differenz zu a2_tr). Wird dann auf die Auftragszeit aufaddiert, damit zu lange Rüsten die Kapazität nicht verkürzt

  a2_vfjobid                integer,                                -- ID vom NC-Fremdsystem "ValueFacturing-Maschinensteuerung (Maschinenfabrik Reinhausen)"

  -- Referenz zum zugeordneten Messprogramm
  a2_msp_id                 integer REFERENCES messprogram ON UPDATE CASCADE
 );


--

/*
+Ressource / Mitarbeiter (Evtl aus Kostenstelle ziehen?)
+Dauer? Belastung pro Tag.
+Mitarbeiterzeit <-> automatische Maschinenlaufzeit
*/

--Vorgangsanordnungen in TABLE anlstr_vorgangsanordnung ( J.1 Projekverwaltung.sql )

--Indizes

  CREATE INDEX ab2_ix_id_not_ende ON ab2 ( a2_ab_ix )      INCLUDE (a2_id) WHERE ( NOT a2_ende ); -- include wegen join
  CREATE INDEX ab2_id_ix_not_ende ON ab2 ( a2_id )         INCLUDE (a2_ab_ix) WHERE ( NOT a2_ende ); -- include wegen join

  CREATE UNIQUE INDEX ab2_ix_opn  ON ab2 ( a2_ab_ix, a2_n) INCLUDE (a2_id);

  CREATE INDEX ab2_makeganttid ON ab2(makeganttopixid(a2_ab_ix, a2_n));--verwendung: Verlinkungen Projektmanagement, Anordnungsbeziehungen

  CREATE INDEX ab2_ks_term ON ab2(a2_ks, a2_at) WHERE a2_interm;
  CREATE INDEX ab2_ks_ende ON ab2(a2_ks, a2_ende);

  CREATE INDEX ab2_termweek ON ab2 (termweek(a2_at));

  CREATE INDEX ab2_a2_at ON ab2(CAST(a2_at AS DATE));
  CREATE INDEX ab2_a2_et ON ab2(CAST(a2_et AS DATE));

  CREATE INDEX ab2_a2_id_a2_o2_id ON ab2(a2_o2_id, a2_id);


--
--- Constraint  #19516
ALTER TABLE ab2
  ADD CONSTRAINT ab2__ausw__requires__dlz
    CHECK((NOT a2_ausw) OR (coalesce( a2_dlz, 0 ) > 0));

-- removes invalid ksaps
CREATE OR REPLACE FUNCTION ab2__b_iu__validate_ksap() RETURNS TRIGGER AS $$
    BEGIN
        IF NOT ( array[new.a2_ksap]::varchar[] && tplanterm.ksv__ksaps__fetch( new.a2_ks ) ) THEN
            new.a2_ksap := NULL;
        END IF;
        --

        RETURN new;
    END $$ LANGUAGE plpgsql;
--
DROP TRIGGER IF EXISTS ab2__b_iu__validate_ksap ON ab2;
CREATE TRIGGER ab2__b_iu__validate_ksap
   BEFORE UPDATE
   OF a2_ks, a2_ksap
   ON ab2
   FOR EACH ROW
   WHEN ( ( old.a2_ks IS DISTINCT FROM new.a2_ks OR old.a2_ksap IS DISTINCT FROM new.a2_ksap ) AND NOT new.a2_ende )
   EXECUTE PROCEDURE ab2__b_iu__validate_ksap();

--
 CREATE OR REPLACE FUNCTION ab2__b_i() RETURNS TRIGGER AS $$
  BEGIN
    --op2
    IF new.a2_o2_id IS NULL THEN
           new.a2_o2_id:=o2_id FROM op2, abk WHERE ab_ix=new.a2_ab_ix AND o2_ix=ab_askix AND o2_n=new.a2_n;
    END IF;
    --
    IF new.a2_n IS NULL THEN
           new.a2_n:=max(a2_n)+10 FROM ab2 WHERE a2_ab_ix=new.a2_ab_ix;
           IF new.a2_n IS NULL THEN
                   new.a2_n:=10;
           END IF;
    END IF;
    --
    RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_i
   BEFORE INSERT
   ON ab2
   FOR EACH ROW
   EXECUTE PROCEDURE ab2__b_i();

 CREATE OR REPLACE FUNCTION ab2__b_iu() RETURNS TRIGGER AS $$
   BEGIN
     new.a2_lgz:=COALESCE(new.a2_lgz, 0);
     new.a2_ul:=COALESCE(new.a2_ul,0);
     --
     IF new.a2_ul<>0 THEN
           new.a2_lgz:=0;
     END IF;
     IF new.a2_ul>100 OR new.a2_ul<0 THEN
           RAISE EXCEPTION '%', lang_text(2062);
     END IF;


     IF (TG_OP ='UPDATE') THEN
       -- Prüfen ob Arbeitsgangtext geändert wurde und Betreff = Erste 50 Zeichen von altem AG-Text entspricht => Betreff neu aus AG-Text bauen
       IF (new.a2_txt IS DISTINCT FROM old.a2_txt) AND (new.a2_txt IS NOT NULL)  AND (old.a2_subject = old.a2_txt::VARCHAR(50)) THEN
          new.a2_subject = new.a2_txt::VARCHAR(50);
       END IF;
     END IF;
     --
     RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_iu
   BEFORE INSERT OR UPDATE
   ON ab2
   FOR EACH ROW
   EXECUTE PROCEDURE ab2__b_iu();

--zeiten in sekunden zurückrechnen, a2_ta
  CREATE OR REPLACE FUNCTION ab2__b_iu__timechange_uf1()
    RETURNS TRIGGER
    AS $$
    DECLARE zeinhr integer;
            zeinhx integer;
    BEGIN
      zeinhr:=new.a2_zeinh_tr;
      zeinhx:=new.a2_zeinh_tx;

      zeinhr:=z_uf FROM zeinh WHERE z_id=zeinhr;
      zeinhx:=z_uf FROM zeinh WHERE z_id=zeinhx;

      new.a2_tr_sek:=new.a2_tr*zeinhr;

      new.a2_th_sek:=new.a2_th*zeinhx;
      new.a2_tn_sek:=new.a2_tn*zeinhx;
      new.a2_tm_sek:=new.a2_tm*zeinhx;

      new.a2_lgz_sek:=new.a2_lgz*86400;     -- 86400 Sekunden = 24 * 60 * 60 Sekunden

        -- zu viel gestempelte Rüstzeit ermitteln.
        IF new.a2_time_stemp__ruest > new.a2_tr_sek / 3600 THEN
            new.a2_time_stemp__ruest_overlap := abs( new.a2_tr_sek / 3600 - new.a2_time_stemp__ruest );
        ELSE
            new.a2_time_stemp__ruest_overlap := 0;
        END IF;

        -- Berechnung gesamte Bearbeitungszeit
        new.a2_ta :=       (-- Rüstzeit
                            coalesce(new.a2_tr_sek, 0)
                            -- + (Hauptzeit + Nebenzeit ) * Verteilzeit %
                            + (
                                (  coalesce(new.a2_th_sek, 0)
                                + coalesce(new.a2_tn_sek, 0)
                                )
                                * (1 + new.a2_tv / 100)
                                )
                                * ifthen( coalesce(ab_st_uf1, 0) = 0, 1, ab_st_uf1 )
                            )
                            / 3600
                      FROM abk
                      WHERE ab_ix = new.a2_ab_ix;

      RETURN new;
    END$$LANGUAGE plpgsql;

  DROP TRIGGER IF EXISTS ab2__b_iu__timechange_uf1 ON ab2;
  CREATE TRIGGER ab2__b_iu__timechange_uf1
    BEFORE INSERT OR UPDATE
    OF a2_zeinh_tr, a2_zeinh_tx,
      a2_tr, a2_th, a2_tn, a2_tm, a2_tv, a2_lgz,
      a2_ab_ix,
      -- a2_ta: bei Änderung Menge im Kopf wird a2_ta -> damit hier neue Berechnung ausgelöst wird
      a2_ta,
      -- a2_time_stemp__ruest: überzogene Rüstzeit erhöht die Auftragszeit
      a2_time_stemp__ruest
    ON ab2
    FOR EACH ROW
  EXECUTE PROCEDURE ab2__b_iu__timechange_uf1();
--

--
 CREATE OR REPLACE FUNCTION ab2__b_iu__ks() RETURNS TRIGGER AS $$
    DECLARE
      _opn    VARCHAR;
      _ksv_record record;
     BEGIN

      SELECT *
      INTO _ksv_record
      FROM ksv
      WHERE ks_abt = new.a2_ks;

      IF ( _ksv_record IS NULL ) THEN
        -- this will fail due to voileted FK constraint for a2_ks -> ks_abt
        RETURN new;

      END IF;

      IF ( _ksv_record.ks_sperr ) THEN
            RAISE EXCEPTION '% "%", ABK: %, AG: %', lang_text(10203), new.a2_ks, new.a2_ab_ix, new.a2_n;
      END IF;

      -- Auswärtskennzeichen in Kostenstelle übernehmen
       new.a2_ausw := _ksv_record.ks_ausw;

       -- Auswärtsbearbeiter aus Kostenstelle, wenn der nicht schon angegeben ist
       IF new.a2_ausw AND new.a2_adkrz IS NULL THEN
            new.a2_adkrz := _ksv_record.ks_krzl;
       END IF;

       -- Prüfung ob AuswärtsArbeitspaket angegeben, wenn das per Sys.Einstellung verlangt wird
       IF new.a2_ausw AND COALESCE( new.a2_aknr, '' ) = '' THEN

           IF TSystem.Settings__GetBool('AWKAT') THEN

             SELECT op_n
             INTO _opn
             FROM opl
             JOIN abk ON ab_askix = op_ix
             WHERE ab_ix = new.a2_ab_ix;

             --"Ein Auswärtsarbeitsgang benötigt ein Arbeitspaket. Stammkarte oder ABK muss angepasst werden. ABK=65300.120, Artikel=bbb
             RAISE EXCEPTION '% % = %.%, KS: %, % = %', lang_text(12655), lang_text(105), new.a2_ab_ix, COALESCE(new.a2_n::VARCHAR,''), new.a2_ks , lang_text(234), COALESCE(_opn,'?');
           ELSE
             new.a2_aknr := ak_nr FROM art WHERE ak_nr = TSystem.Settings__Get('AUSWAVORVORGABE');
           END IF;

       END IF;
      --
      RETURN new;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_iu__ks
   BEFORE INSERT OR UPDATE
   OF a2_ks
   ON ab2
   FOR EACH ROW
   EXECUTE PROCEDURE ab2__b_iu__ks();
--
CREATE OR REPLACE FUNCTION ab2__b_90_iu() RETURNS TRIGGER AS $$
  BEGIN
    -- im allgemeinen Trigger, falls in irgendeinem Trigger zB da Menge < Menge geliefert das umgesetzt wird und es kein OF gibt!
    IF new.a2_ende IS true THEN
      new.a2_ende_dat := coalesce(new.a2_ende_dat, now());
    ELSE
      new.a2_ende_dat := null;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_90_iu
    BEFORE INSERT OR UPDATE
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__b_90_iu();

--
 CREATE OR REPLACE FUNCTION ab2__a_i() RETURNS TRIGGER AS $$
   DECLARE todotxt TEXT;
   BEGIN
     --Arbeitsaufgabe aus ASK
     SELECT o2_todotxt INTO todotxt
       FROM op2
      WHERE o2_id = new.a2_o2_id;

     IF todotxt IS NOT NULL THEN
        INSERT INTO ab2_wkstplan_response (awpr_a2_id, awpr_text) VALUES (new.a2_id, todotxt);
     END IF;
     --

     -- rückgabeparameter aus op2
     INSERT INTO ab2_resultparam (a2r_a2_id, a2r_vname, a2r_vbez, a2r_type, a2r_force, a2r_txt)
          SELECT new.a2_id, o2r_vname, o2r_vbez, o2r_type, o2r_force, o2r_txt
            FROM op2_resultparam
           WHERE o2r_o2_id = new.a2_o2_id;

     -- ABK bei neuem AG automatisch wieder öffnen
     IF NOT new.a2_ende THEN
        UPDATE abk SET ab_done = false WHERE ab_done AND ab_ix = new.a2_ab_ix;
     END IF;

     RETURN new;
   END $$ LANGUAGE plpgsql;

    CREATE TRIGGER ab2__a_i
      AFTER INSERT
      ON ab2
      FOR EACH ROW
      EXECUTE PROCEDURE ab2__a_i();
--

--
CREATE OR REPLACE FUNCTION ab2__a_i__ab2_wkstplan() RETURNS TRIGGER AS $$
  BEGIN
    -- Anlegen Datensatz in ab2_wkstplan
       INSERT INTO ab2_wkstplan
                   (a2w_a2_id
                    )
            VALUES (new.a2_id
                    )
       ;

    RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  DROP TRIGGER IF EXISTS ab2__a_i__ab2_wkstplan ON ab2;
  CREATE TRIGGER ab2__a_i__ab2_wkstplan
    AFTER INSERT
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__a_i__ab2_wkstplan();
--

 -- Legt beim Insert die Default-Parameter für Arbeistgänge an
 CREATE TRIGGER ab2__a_i__create_autoparams
     AFTER INSERT
     ON ab2
     FOR EACH ROW
     EXECUTE PROCEDURE TRecnoParam.CreateAutoParams();
--

 ---------------------------
 CREATE OR REPLACE FUNCTION ab2__a_u__a2_rckmeld() RETURNS TRIGGER AS $$
   BEGIN
     UPDATE abk SET ab_rckmeld = true WHERE ab_ix = new.a2_ab_ix AND NOT ab_rckmeld;
     RETURN new;
   END $$ LANGUAGE plpgsql;

   CREATE TRIGGER ab2__a_u__a2_rckmeld
     AFTER UPDATE
     OF a2_rckmeld
     ON ab2
     FOR EACH ROW
     WHEN (new.a2_rckmeld AND NOT old.a2_rckmeld)
     EXECUTE PROCEDURE ab2__a_u__a2_rckmeld();
--

--
CREATE OR REPLACE FUNCTION ab2__a_iu__a2_ende() RETURNS TRIGGER AS $$
  BEGIN
    IF new.a2_ende THEN
        -- Vorhergehende nicht planrelevante AGe automatisch ebenfalls als beendet setzen. Nicht planrelevante AGe werden nicht einterminiert und sollten mit dem aktuellen AG schon bearbeitet/beachtet sein, daher hier das automatische beenden.
        IF NOT old.a2_ende THEN
            WITH a AS (
                SELECT a2_id
                  FROM ab2 JOIN abk ON a2_ab_ix = ab_ix
                  LEFT JOIN ab2_wkstplan ON a2w_a2_id = a2_id
                  JOIN ksv ON ks_abt = coalesce( ( scheduling.resource__translate__resource_id__to__ksvba__shorthand( a2w_resource_id_main_fix ) ).ksb_ks_abt, a2_ks )
                 WHERE a2_ab_ix = new.a2_ab_ix  -- Alle AGe der gleichen ABK, ...
                   AND NOT TSystem.ENUM_ContainsValue(ab_stat, 'AM') -- Änderungsmanagement: VorAGs sind organisatorisch und dürfen NICHT beendet werden => https://redmine.prodat-sql.de/issues/22152
                   AND a2_n < new.a2_n          -- ... welche vor dem aktuellen AG liegen, ...
                   AND NOT ks_plan              -- ... nicht planungsrelevat und ...
                   AND NOT a2_ende              -- ... noch nicht beendet sind.
            )
            UPDATE ab2 AS b
               SET a2_ende = true
              FROM a
             WHERE b.a2_id = a.a2_id
            ;
        END IF;

        -- ABK schliessen wenn alle AG beendet sind.
        IF NOT EXISTS(SELECT true FROM ab2 WHERE NOT a2_ende AND a2_ab_ix = new.a2_ab_ix) THEN
            UPDATE abk SET ab_done=TRUE WHERE NOT ab_done AND ab_ix=new.a2_ab_ix;--Material(auftg) ist hier außen vor. Beachte Set-Artikel: hier schließt die letzte Materialposition die ABK, wenn diese keine AG hat (auftg__a_u__agdone_close)
        END IF;

        -- AG beendet. Termin ABK "Terminiert" aktualisieren
        PERFORM scheduling.abk__at_et__from__ab2__sync( new.a2_ab_ix ); -- darin Logging

        -- Abschluss aller zum AG gehörigen Messungen, sofern dynamische Einstellung gesetzt
        IF tsystem.settings__getbool( 'MESSWERTE_DEFINITIV_NACH_AG_ENDE' ) THEN
            UPDATE oplpm_mw SET mw_definitiv = true WHERE
                    mw_pm_id IN (
                        SELECT pm_id FROM oplpm_data WHERE pm_a2_id = new.a2_id
                    )
                AND NOT mw_definitiv;
        END IF;
    END IF;

    IF tg_op = 'UPDATE' THEN -- OF a2_ende
        IF NOT new.a2_ende AND old.a2_ende THEN -- Arbeitsgang wird wieder geöffnet
            UPDATE abk SET ab_done = false WHERE ab_done AND ab_ix = new.a2_ab_ix; -- ABK wieder öffnen
        END IF;

        -- Eintrag von Ausschuss in ABK-Eigenschaften. vgl. bdea__a_20_iud_ausschuss
        IF new.a2_ende IS DISTINCT FROM old.a2_ende THEN -- nur bei Statuswechsel neu berechnen
            -- Nur wenn Auftragszeiten oder Auswärstvergaben existieren, muss Ausschuss berechnet/geschrieben werden.
            IF   EXISTS(SELECT true FROM bdea WHERE ba_ix = new.a2_ab_ix AND ba_op = new.a2_n)
              OR EXISTS(SELECT true FROM ldsdok WHERE ld_a2_id = new.a2_id)
            THEN
               PERFORM tabk.set_abkrecno_unterproduktion(new.a2_ab_ix);
            END IF;
        END IF;

        -- Terminierung für beendete AG entfernen
        -- Wenn ABK beendet wird, dann beendet der Trigger "abk__a_60_u__AGbeenden" auch die zugehörigen Arbeitsgänge. Somit müssen in den ABK-Triggern die Terminierungen der zugehörigen AGe nicht extra gelöscht werden.
        -- AND NOT a2_ende => JOIN verbraucht recht viel Zeit, so würden aber auch vergangene Timeslots geladen, daher müssen diese entfernt werden bei a2_ende siehe PT.TPlantafelContainerDataSource_Core
        IF new.a2_ende AND NOT old.a2_ende THEN
           DELETE FROM scheduling.resource_timeline WHERE ti_a2_id = new.a2_id;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__a_iu__a2_ende
    AFTER INSERT OR UPDATE
    OF a2_ende
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__a_iu__a2_ende();
--

  CREATE OR REPLACE FUNCTION ab2__a_50_u__a2_interm_inplantaf() RETURNS TRIGGER AS $$
      DECLARE _ab_inplantaf boolean;
      BEGIN
        -- wenn ein AG einterminiert ist, ist es interm
        _ab_inplantaf := bool_or(a2_interm) FROM ab2 WHERE a2_ab_ix = new.a2_ab_ix AND NOT a2_ende;
        UPDATE abk SET ab_inplantaf = _ab_inplantaf WHERE ab_ix = new.a2_ab_ix AND ab_inplantaf <> _ab_inplantaf;
        RETURN new;
      END $$ LANGUAGE plpgsql;
      --
      DROP TRIGGER IF EXISTS ab2__a_50_u__a2_interm_inplantaf ON ab2;
      CREATE TRIGGER ab2__a_50_u__a2_interm_inplantaf
        AFTER UPDATE
        OF a2_at, a2_et, a2_interm --at und et, da interm nur im trigger gesetzt wird und somit nicht nochmal aufgerufen wird
        ON ab2
        FOR EACH ROW
        EXECUTE PROCEDURE ab2__a_50_u__a2_interm_inplantaf(); -- TODO das ist während der Terminierung MIST; da es pro Datensatz passiert

--
 CREATE OR REPLACE FUNCTION ab2__a_iu__a2_n() RETURNS TRIGGER AS $$
  DECLARE planpool BOOLEAN;
          r        RECORD;
          arbpnum  INTEGER;
  BEGIN
    -- Umnummerieren des Arbeitsgangs, nachziehen in Anfrage
    UPDATE anfart SET aart_o2_n = new.a2_n WHERE aart_a2_id = new.a2_id; -- Ja, das Feld ist wirklich aart_o2_n.
    UPDATE bdea SET ba_op = new.a2_n WHERE ba_ix=new.a2_ab_ix AND ba_op=old.a2_n;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__a_iu__a2_n
    AFTER UPDATE OF a2_n
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__a_iu__a2_n();
--

--
 CREATE OR REPLACE FUNCTION ab2__a_u_checkResultparam() RETURNS TRIGGER AS $$
  DECLARE abk_ab2_with_rm VARCHAR;
  BEGIN
    IF EXISTS (SELECT TRUE FROM ab2_resultparam
               WHERE new.a2_id=a2r_a2_id
                 AND a2r_force
                 AND COALESCE(a2r_value,'')='')
        AND NOT (SELECT ab_storno FROM abk WHERE new.a2_ab_ix=ab_ix)
    THEN
        SELECT array_to_string(ARRAY( -- Stringliste der fehlenden AG mit RM bauen
            SELECT a2_n FROM ab2
            WHERE a2_ab_ix=new.a2_ab_ix
              AND EXISTS(SELECT TRUE FROM ab2_resultparam WHERE a2_id=a2r_a2_id AND a2r_force AND COALESCE(a2r_value,'')='')
            ORDER BY a2_n) -- SELECT a2_n ENDE
        , ', ') -- array_to_string Separator
        INTO abk_ab2_with_rm;

        RAISE EXCEPTION E'% % % \n', lang_text(16095), E'\n\n ABK: '||new.a2_ab_ix, E'\n AG: '||COALESCE(abk_ab2_with_rm, ''); --Erforderliche Rückgabeparameter zum Beenden des Arbeitsgangs nicht angegeben!

    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__a_u_checkResultparam
    AFTER UPDATE
    OF a2_ende
    ON ab2
    FOR EACH ROW
    WHEN (new.a2_ende AND NOT old.a2_ende)
    EXECUTE PROCEDURE ab2__a_u_checkResultparam();
--

--
 CREATE OR REPLACE FUNCTION ab2__a_u_a2_buch() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (NOT new.a2_ausw AND new.a2_buch AND NOT old.a2_buch)
    IF NOT EXISTS(SELECT true FROM nk2 WHERE n2_a2_id = new.a2_id) THEN -- keine NK für den AG vorhanden
        IF EXISTS(SELECT true FROM bdea WHERE ba_ix = new.a2_ab_ix AND ba_op = new.a2_n) -- aber Auftragszeiten
            OR EXISTS(SELECT true FROM rm WHERE r_a2_id = new.a2_id) -- oder Rückmeldungen
        THEN
            RAISE EXCEPTION '%', lang_text(16477) || E'\n\n' || lang_text(105) || ': ' || new.a2_ab_ix || E'\n' || lang_text(160) || ': ' || new.a2_n;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__a_u_a2_buch
    AFTER UPDATE
    OF a2_buch
    ON ab2
    FOR EACH ROW
    WHEN (NOT new.a2_ausw AND new.a2_buch AND NOT old.a2_buch)
    EXECUTE PROCEDURE ab2__a_u_a2_buch();
--
 CREATE OR REPLACE FUNCTION ab2__b_iu__ks__ausw__requires__dlz() RETURNS TRIGGER AS $$
   BEGIN
     new.a2_dlz := ( SELECT coalesce( ks_dlz, 0 ) FROM ksv WHERE ks_abt = new.a2_ks );
     --
     RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_iu__ks__ausw__requires__dlz
   BEFORE INSERT OR UPDATE
   OF a2_ks, a2_ausw
   ON ab2
   FOR EACH ROW
   WHEN ( new.a2_ausw AND coalesce( new.a2_dlz, 0 ) = 0 )
   EXECUTE PROCEDURE ab2__b_iu__ks__ausw__requires__dlz();
--
 CREATE OR REPLACE FUNCTION ab2__b_d() RETURNS TRIGGER AS $$
  BEGIN
    --AG loeschen sperren, wenn es Stempelzeiten auf den Arbeitsgang gibt
    IF EXISTS(SELECT true FROM bdea WHERE ba_ix = old.a2_ab_ix AND ba_op = old.a2_n AND ba_ks = old.a2_ks) THEN
        RAISE EXCEPTION '%', lang_text(12501);
    END IF;

    DELETE FROM ab2_wkstplan WHERE a2w_a2_id=old.a2_id;
    DELETE FROM ab2_resultparam WHERE a2r_a2_id=old.a2_id AND a2r_value IS NULL;
    DELETE FROM anlstr_vorgangsanordnung WHERE ava_p_a2id=old.a2_id;
    DELETE FROM anlstr_vorgangsanordnung WHERE ava_c_a2id=old.a2_id;

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2__b_d
    BEFORE DELETE
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__b_d();
--

-- Überhänge Blockieren
-- Ein INSERT-Trigger ist nicht notwendig, da bei noch NICHT exsitierenden AG keine Stempelungen existieren können und somit Spalte a2_time_stemp NULL bzw. 0 sein muss.
-- Ein Delete-Trigger ist nicht notwendig, da vor dem Löschen des AG alle zugehörigen Stempelungen gelöscht werden müssen.
CREATE OR REPLACE FUNCTION ab2__a_u__blocktimes() RETURNS TRIGGER AS $$
  BEGIN
      -- Stempelung aktualisiert.
      IF old.a2_time_stemp IS DISTINCT FROM new.a2_time_stemp THEN
          -- AG is terminiert.
          IF EXISTS( SELECT true FROM scheduling.resource_timeline WHERE ti_a2_id = new.a2_id AND ti_type = 'task' ) THEN
              PERFORM scheduling.resource_timeline__ab2__retermination_fragmented( new );
          END IF;
      END IF;

      RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  DROP TRIGGER IF EXISTS ab2__a_u__blocktimes ON ab2;
  CREATE TRIGGER ab2__a_u__blocktimes
    AFTER UPDATE OF a2_time_stemp
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__a_u__blocktimes();
--


-- #7125 Prüfprotokoll elektronisch
    CREATE OR REPLACE FUNCTION TABK.oplpm_data__create_from__oplpm__for__abk(in_a2_id INTEGER) RETURNS BOOLEAN AS $$
      DECLARE r RECORD;
    BEGIN
        SELECT COALESCE(ab_st_uf1_soll, ab_st_uf1) AS ab_st_uf1, a2_o2_id, EXISTS(SELECT true FROM oplpm WHERE pm_op2_id=a2_o2_id) AS pmexists INTO r FROM ab2 JOIN abk ON ab_ix=a2_ab_ix WHERE a2_id=in_a2_id; --hole Anzahl Teile sowie ob Prüfmerkmale existieren

        IF NOT r.pmexists THEN --Kein prüfmerkmal vorhanden, weg
           RETURN false;
        END IF;

        --

        DROP TABLE IF EXISTS oplpm_tmp;

        CREATE TEMP TABLE oplpm_tmp (LIKE oplpm_data);

        -- umkopieren in temporäre Tabelle
        INSERT INTO oplpm_tmp SELECT * FROM oplpm WHERE pm_op2_id = r.a2_o2_id
                                    --nächste Zeile: ausschliessen von Merkmalen, welche bereits kopiert wurden, Entscheidung anhand MerkmalNr
                                    AND NOT EXISTS(SELECT true FROM oplpm_data WHERE oplpm_data.pm_op2_id = r.a2_o2_id AND oplpm_data.pm_a2_id = in_a2_id AND oplpm_data.pm_pmnr = oplpm.pm_pmnr);
        -- in temporärer Tabelle ID's ändern
        UPDATE oplpm_tmp SET
          pm_id = nextval('oplpm_data_pm_id_seq'),
          dbrid = nextval('db_id_seq'),
          pm_a2_id = in_a2_id,
       --   pm_anzpruef = r.ab_st_uf1,      -- gelöscht
          insert_by = current_user,
          insert_date = current_date,
          modified_by = NULL,
          modified_date = NULL;
        /*
        #11911  [KB.Proj.MP] Feld PM_ADDVORGABE entfernen &
        #11927  [KB.Proj.MP] Feld ain_pruef_lng entfernen
        -- modifikation Nennmaß um Zusatz aus Artikelstamm
        UPDATE oplpm_tmp SET pm_nenn = AsNumeric(pm_nenn, True)+AsNumeric(artinfo.ain_pruef_lng, True)
            FROM ab2 JOIN abk ON a2_ab_ix=ab_ix JOIN opl ON op_ix=ab_askix JOIN artinfo ON artinfo.ain_ak_nr = opl.op_n WHERE a2_id=pm_a2_id AND AsNumeric(pm_nenn) IS NOT NULL AND pm_addvorgabe;
        */
        -- umkopieren in reale Tabelle
        INSERT INTO oplpm_data SELECT * FROM oplpm_tmp;
        -- temporäre Tabelle löschen
        DROP TABLE IF EXISTS oplpm_tmp;
        RETURN true;
     END $$ LANGUAGE plpgsql VOLATILE;
--

    -- Triggerfunktion ab2__a_iu_pruefmerkmale
    CREATE OR REPLACE FUNCTION ab2__a_i_pruefmerkmale() RETURNS TRIGGER AS $$
    BEGIN
      PERFORM TABK.oplpm_data__create_from__oplpm__for__abk(new.a2_id);
      RETURN new;
    END $$ LANGUAGE plpgsql VOLATILE;

    CREATE TRIGGER ab2__a_i_pruefmerkmale
    AFTER INSERT
    ON ab2
    FOR EACH ROW
    EXECUTE PROCEDURE ab2__a_i_pruefmerkmale();
--


CREATE OR REPLACE FUNCTION ab2_getvabg_ende(a2id INTEGER) RETURNS BOOL AS $$
  DECLARE r RECORD;
          b BOOLEAN;
  BEGIN
    b:=TRUE;
    --Vorgangsanordnung auf Basis von Beziehung
    FOR r IN SELECT a2_ende FROM public.anlstr_vorgangsanordnung JOIN public.ab2 ON a2_id=ava_p_a2id WHERE ava_c_a2id=a2id LOOP
           b:=b AND r.a2_ende;
    END LOOP;
    --keine Beziehung, dann klassisch hierarisch
    IF NOT FOUND THEN
           b:=av.a2_ende FROM ab2 av /*vorarbeitsgang*/ JOIN ab2 a2 /*jetziger arbeitsgang*/ ON a2.a2_ab_ix=av.a2_ab_ix WHERE a2.a2_id=a2id AND av.a2_n<a2.a2_n ORDER BY av.a2_n DESC LIMIT 1;
    END IF;
    --
    RETURN COALESCE(b, true);
  END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tartikel.art__ak_copy_exact__parent__is(IN _ab2 ab2) RETURNS boolean
  AS $$
    SELECT tartikel.art__ak_copy_exact__is(ab_ap_nr) FROM abk WHERE ab_ix = _ab2.a2_ab_ix;
  $$ LANGUAGE sql STABLE PARALLEL SAFE;

CREATE OR REPLACE FUNCTION tartikel.art__ak_copy_exact__parent__is(IN _auftg auftg) RETURNS boolean
  AS $$
    SELECT coalesce(
                    (SELECT tartikel.art__ak_copy_exact__is(ab_ap_nr) FROM abk WHERE ab_ix = _auftg.ag_parentabk),
                    false
                    );
  $$ LANGUAGE sql STABLE PARALLEL SAFE;


-- auftragsbegleitende Artikel der Fertigung an bestimmten AG
CREATE TABLE ab2ba (
  a2ba_id              SERIAL PRIMARY KEY,
  a2ba_ak_nr           VARCHAR(40) NOT NULL,
  a2ba_noz_id          INTEGER REFERENCES normzert,  -- die qs_ident in a2ba_ak_nr
  a2ba_a2_id           INTEGER REFERENCES ab2 ON UPDATE CASCADE ON DELETE CASCADE,

    -- Der ABK-Index enthält die ABK, in welcher der Auftragsbegleitende Artikel produziert wird.
    -- Er entspricht NICHT dem aktuellen ABK-Index des zugewiesenen AGs.
  a2ba_ab_ix           INTEGER REFERENCES abk ON UPDATE CASCADE,
  a2ba_txt             TEXT,
  a2batxt_rtf          TEXT
);
--

-- Indizes
  CREATE INDEX ab2ba_a2ba_ab_ix         ON ab2ba (a2ba_ab_ix) WHERE a2ba_ab_ix IS NOT NULL;
  CREATE INDEX ab2ba_a2ba_a2_id         ON ab2ba (a2ba_a2_id);
  CREATE INDEX ab2ba_a2ba_ak_nr         ON ab2ba (a2ba_ak_nr);
  CREATE INDEX ab2ba_a2ba_ak_nr_like    ON ab2ba (a2ba_ak_nr varchar_pattern_ops);
--

-- Aktuelle ABK des auftragsbegleitenden Artikels verknüpfen. #8423
-- vgl. abk__a_iu__ab2ba
CREATE OR REPLACE FUNCTION ab2ba__b10_iu__abk() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.a2ba_ab_ix IS NULL) -- Anwendereingaben berücksichtigen. Bei Artikelwechsel kann hier nicht automatisch reagiert werden.
    new.a2ba_ab_ix:=
        ab_ix FROM abk
          -- LEFT JOIN ldsdok ON ld_abk = ab_ix
          -- LEFT JOIN anl ON ab_tablename = 'anl' AND anl.dbrid = ab_dbrid -- ggf. einfach an an_abix für Projekt-Artikel
        WHERE NOT ab_done               -- offene ABK
          AND ab_ap_nr = new.a2ba_ak_nr -- des auftragsbegleitenden Artikels
          AND ab_tablename = 'anl' -- #11271 - Nur Werkzeugbau aus Projekten berücksichtigen.
        ORDER BY ab_et, ab_ix -- nach ältestem geplanten Termin der Fertigstellung (NULLS FIRST? ungeplante bevorzugen?)
        LIMIT 1;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2ba__b10_iu__abk
    BEFORE INSERT OR UPDATE
    OF a2ba_ak_nr
    ON ab2ba
    FOR EACH ROW
    WHEN (new.a2ba_ab_ix IS NULL) -- Anwendereingaben berücksichtigen. Bei Artikelwechsel kann hier nicht automatisch reagiert werden.
    EXECUTE PROCEDURE ab2ba__b10_iu__abk();
--


/*Rückgabe des Nutzers beim Stempeln aus AVOR-Vorgaben*/
CREATE TABLE ab2_resultparam
  (a2r_id               SERIAL,
   a2r_a2_id            INTEGER NOT NULL REFERENCES ab2 ON UPDATE CASCADE ON DELETE RESTRICT,
   a2r_vname            VARCHAR(20),                    --Paremetername
   a2r_vbez             VARCHAR(50),                    --Beschreibung/Name Ergebnis, zB Dachneigung
   a2r_type             INTEGER,                        --feldtyp (daum, zahl etc)
   a2r_force            BOOL NOT NULL DEFAULT FALSE,    --zwangsfeld
   a2r_txt              TEXT,                           --hinweis/beschreibung aus vorgabe ASK
   a2r_value            VARCHAR(10000),                 --ergebnis
   a2r_bem              TEXT                            --bemerkung zum Ergebnis
  );


CREATE TABLE ab2_resultparam_log
  ( ab2rl_id                    SERIAL PRIMARY KEY,
    ab2rl_time                  TIMESTAMP(0) WITHOUT TIME ZONE DEFAULT currenttime(), --zeitpunkt (insert und modified date haben nur datum, keine uhrzeit)
    ab2rl_a2r_id                INTEGER,        --a2r = Felder der ab2_resultparam
    ab2rl_a2r_a2_id             INTEGER,        --Ref auf ab2
    ab2rl_a2r_vname_old         VARCHAR(20),    --Paremetername
    ab2rl_a2r_vname_new         VARCHAR(20),    --
    ab2rl_a2r_vbez_old          VARCHAR(50),    --Beschreibung/Name Ergebnis, zB Dachneigung
    ab2rl_a2r_vbez_new          VARCHAR(50),    --
    ab2rl_a2r_type_old          INTEGER,        --feldtyp (daum, zahl etc)
    ab2rl_a2r_type_new          INTEGER,        --
    ab2rl_a2r_force_old         BOOL,           --zwangsfeld
    ab2rl_a2r_force_new         BOOL,           --
    ab2rl_a2r_txt_old           TEXT,           --hinweis/beschreibung aus vorgabe ASK
    ab2rl_a2r_txt_new           TEXT,           --
    ab2rl_a2r_value_old         VARCHAR(10000), --ergebnis
    ab2rl_a2r_value_new         VARCHAR(10000), --
    ab2rl_a2r_bem_old           TEXT,           --bemerkung zum Ergebnis
    ab2rl_a2r_bem_new           TEXT            --
  );

CREATE OR REPLACE FUNCTION ab2_resultparam__a_u_log() RETURNS TRIGGER AS $$
 DECLARE
    a2r_vname_old       VARCHAR;        --Paremetername
    a2r_vname_new       VARCHAR;
    a2r_vbez_old        VARCHAR;        --Beschreibung/Name Ergebnis, zB Dachneigung
    a2r_vbez_new        VARCHAR;
    a2r_type_old        NUMERIC;        --feldtyp (daum, zahl etc)
    a2r_type_new        NUMERIC;
    a2r_force_old       BOOL;           --zwangsfeld
    a2r_force_new       BOOL;
    a2r_txt_old         VARCHAR;        --hinweis/beschreibung aus vorgabe ASK
    a2r_txt_new         VARCHAR;
    a2r_value_old       VARCHAR;        --ergebnis
    a2r_value_new       VARCHAR;
    a2r_bem_old         VARCHAR;        --bemerkung zum Ergebnis
    a2r_bem_new         VARCHAR;
    doit                BOOL;

 BEGIN
  doit:=FALSE;
  --
  IF new.a2r_vname IS DISTINCT FROM old.a2r_vname THEN
    doit:=TRUE;
    a2r_vname_old:=old.a2r_vname;
    a2r_vname_new:=new.a2r_vname;
  END IF;
  --
  IF new.a2r_vbez IS DISTINCT FROM old.a2r_vbez THEN
    doit:=TRUE;
    a2r_vbez_old:=old.a2r_vbez;
    a2r_vbez_new:=new.a2r_vbez;
  END IF;
  --
  IF new.a2r_type IS DISTINCT FROM old.a2r_type THEN
    doit:=TRUE;
    a2r_type_old:=old.a2r_type;
    a2r_type_new:=new.a2r_type;
  END IF;
  --
  IF new.a2r_force IS DISTINCT FROM old.a2r_force THEN
    doit:=TRUE;
    a2r_force_old:=old.a2r_force;
    a2r_force_new:=new.a2r_force;
  END IF;
  --
  IF new.a2r_txt IS DISTINCT FROM old.a2r_txt THEN
    doit:=TRUE;
    a2r_txt_old:=old.a2r_txt;
    a2r_txt_new:=new.a2r_txt;
  END IF;
  --
  IF new.a2r_value IS DISTINCT FROM old.a2r_value THEN
    doit:=TRUE;
    a2r_value_old:=old.a2r_value;
    a2r_value_new:=new.a2r_value;
  END IF;
  --
  IF new.a2r_bem IS DISTINCT FROM old.a2r_bem THEN
    doit:=TRUE;
    a2r_bem_old:=old.a2r_bem;
    a2r_bem_new:=new.a2r_bem;
  END IF;
  --
  --falls aenderung, dann eintragen in log-tabelle
  IF doit THEN
    INSERT INTO ab2_resultparam_log (
    ab2rl_a2r_id, ab2rl_a2r_a2_id,
    ab2rl_a2r_vname_old, ab2rl_a2r_vname_new,
    ab2rl_a2r_vbez_old, ab2rl_a2r_vbez_new,
    ab2rl_a2r_type_old, ab2rl_a2r_type_new,
    ab2rl_a2r_force_old, ab2rl_a2r_force_new,
    ab2rl_a2r_txt_old, ab2rl_a2r_txt_new,
    ab2rl_a2r_value_old, ab2rl_a2r_value_new,
    ab2rl_a2r_bem_old, ab2rl_a2r_bem_new
    )
    VALUES (
    new.a2r_id, new.a2r_a2_id,
    a2r_vname_old, a2r_vname_new,
    a2r_vbez_old, a2r_vbez_new,
    a2r_type_old, a2r_type_new,
    a2r_force_old, a2r_force_new,
    a2r_txt_old, a2r_txt_new,
    a2r_value_old, a2r_value_new,
    a2r_bem_old, a2r_bem_new
    );
  END IF;
  --
  RETURN new;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER ab2_resultparam__a_u_log
 AFTER UPDATE
 ON ab2_resultparam
 FOR EACH ROW
 EXECUTE PROCEDURE ab2_resultparam__a_u_log();


--für Werkstattplanung

CREATE TABLE ab2_wkstplan (
  a2w_id                      serial PRIMARY KEY,
  a2w_a2_id                   integer REFERENCES ab2 ON UPDATE CASCADE ON DELETE CASCADE,
  a2w_resource_id_main_terminated integer,        -- Wenn tatsächlich terminiert die Resource auf welcher terminiert ist.
  a2w_resource_ks_abt_main_terminated varchar(9),
  a2w_resource_id_main_fix    integer,            -- resource auf die zwangsweise terminiert werden soll! kann bis zur nächsten terminierung <> terminated sein
  a2w_resource_ks_abt_main_fix varchar(9),

  -- Folgende 4 Felder nur aus Abwärtskompatibiliät Beachte ab2__at_et__from__resource_timeline__sync sowie ab2_wkstplan__b_u
  a2w_planweek                 integer,--Woche in der der Auftrag eingeplant wurde (termweek a2_at)
  a2w_endweek                  integer,--Woche in der der Auftrag eingeplant wurde
  a2w_oks                      varchar(9),  -- ab2_wkstplan__b_10_u__resource => new.a2w_oks := new.a2w_resource_ks_abt_main_terminated := ksb_ks_abt;
  a2w_ks                       varchar(36), -- ab2_wkstplan__b_10_u__resource
  a2w_ks_ba_babz               varchar(25), -- ksb_ks_ba_babz
  --

  -- TODO DROP
  /*
  --a2w_wstaskinstance   integer REFERENCES wstaskinstance(id) ON UPDATE CASCADE ON DELETE CASCADE,
    a2w_pat                   timestamp(0) without time zone,
    a2w_pet                   timestamp(0) without time zone,
    a2w_ende                  boolean NOT NULL DEFAULT FALSE,
  */
  a2w_prio                  integer, --DEFAULT 5 (mandatec)

  a2w_stukorr                 numeric,
  -- NT a2w_stukorr_maschautota numeric,
  a2w_marked                  integer NOT null DEFAULT 1,--Farbliche Hervorhebung, bei -1 SplitAG
  a2w_changed                 boolean DEFAULT FALSE,--Änderung
  a2w_ncdone                  boolean DEFAULT FALSE,
  a2w_ncby                    varchar(100),
  a2w_underline               boolean DEFAULT FALSE,
  a2w_underlineby             varchar(100),
  a2w_looked                  varchar []--welche nutzer haben die Änderungen schon gesehen
);

-- Indizes
    -- CREATE INDEX ab2_wkstplan_planweek ON ab2_wkstplan (a2w_planweek);
    -- CREATE INDEX ab2_wkstplan_endweek ON ab2_wkstplan (a2w_endweek);
    -- CREATE INDEX ab2_wkstplan_pat ON ab2_wkstplan (a2w_pat);
    CREATE INDEX ab2_wkstplan_a2_id ON ab2_wkstplan (a2w_a2_id);
    CREATE INDEX ab2_wkstplan_resource_id_main_terminated ON ab2_wkstplan (a2w_resource_id_main_terminated) WHERE a2w_resource_id_main_terminated IS NOT null;
    -- CREATE INDEX ab2_wkstplan_a2_ks ON ab2_wkstplan (a2w_ks, a2w_ende);
    -- CREATE INDEX ab2_wkstplan_a2_oks ON ab2_wkstplan (a2w_oks) WHERE NOT a2w_ende;
    -- CREATE INDEX ab2_wkstplan_ende ON ab2_wkstplan (a2w_ende);
--

-- TRIGGER: ab2_wkstplan__b_10_u__resource suche in files

CREATE OR REPLACE FUNCTION ab2_wkstplan__b_50_u__a2w_changed() RETURNS TRIGGER AS $$
  DECLARE _r record;
  BEGIN
    --a2w_changed nur innerhalb der ersten 3 wochen anzeigen
    IF new.a2w_changed THEN
       IF (SELECT termweek(a2_at) FROM ab2 WHERE a2_id = new.a2w_a2_id) > termweek(current_date + 21) THEN
          new.a2w_changed := false;
       END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  DROP TRIGGER IF EXISTS ab2_wkstplan__b_50_u__a2w_changed ON ab2_wkstplan;
  CREATE TRIGGER ab2_wkstplan__b_50_u__a2w_changed
    BEFORE UPDATE OF a2w_stukorr, a2w_marked, a2w_ncdone, a2w_underline
    ON ab2_wkstplan
    FOR EACH ROW
    EXECUTE PROCEDURE ab2_wkstplan__b_50_u__a2w_changed();
--

CREATE OR REPLACE FUNCTION tartikel.art__ak_copy_exact__parent__is(IN _ab2_wkstplan ab2_wkstplan) RETURNS boolean
  AS $$
    SELECT     tartikel.art__ak_copy_exact__is(ab_ap_nr)
           -- Die Resource darf nicht von der AVOR=>AB2 (ab2 ist CE geschützt, daher reicht die Prüfung ab2) abweichen!
           AND a2_ks IS DISTINCT FROM _ab2_wkstplan.a2w_oks
      FROM ab2 JOIN abk ON ab_ix = a2_ab_ix
     WHERE a2_id = _ab2_wkstplan.a2w_a2_id
           -- austerminiert komplett
       AND _ab2_wkstplan.a2w_oks IS NOT null;
  $$ LANGUAGE sql STABLE PARALLEL SAFE;


-- TODO DROP?
CREATE TABLE ab2_wkstplan_stu--Stundenkorrekturen für Arbeitsgänge
     (a2ws_id               serial PRIMARY KEY,
      a2ws_a2w_id           integer NOT NULL REFERENCES ab2_wkstplan ON UPDATE CASCADE ON DELETE CASCADE,
      a2ws_week             integer NOT NULL,
      a2ws_stu              numeric(10,2),                  --Stundenkorrektur: Begrenzung auf Stunden
      a2ws_stu_maschauto_ta numeric(8,2)                    --
     );


 CREATE INDEX a2ws_a2w_id ON ab2_wkstplan_stu (a2ws_a2w_id);

 CREATE FUNCTION ab2_wkstplan_stu__a_iu__delete_ifnull() RETURNS TRIGGER AS $$
  BEGIN
   DELETE FROM ab2_wkstplan_stu WHERE a2ws_id=new.a2ws_id;
   RETURN NULL;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER ab2_wkstplan_stu__a_iu
   AFTER INSERT OR UPDATE
   ON ab2_wkstplan_stu
   FOR EACH ROW
   WHEN (new.a2ws_stu IS NULL AND new.a2ws_stu_maschauto_ta IS NULL)
   EXECUTE PROCEDURE ab2_wkstplan_stu__a_iu__delete_ifnull();
--

--


CREATE TABLE wkst_kapbelast --Kapazität und Belastung einer Kostenstelle in einer Woche
 (wb_id                 SERIAL PRIMARY KEY,
  -- todo auf resource id
  wb_ks                 VARCHAR NOT NULL,               -- Arbeitsplatz, entspricht ksb_ks_shorthand
  wb_oks                VARCHAR NOT NULL,               -- Kostenstelle, entspricht ks_abt
  --
  wb_comment            TEXT,
  wb_week               VARCHAR,
  wb_weekfix            BOOL NOT NULL DEFAULT false,    --woche fixiert, nur Kapazitätsberechtigte dürfen noch Änderungen vornehmen.
  wb_weekfixauto        BOOL NOT NULL DEFAULT false,    --woche wurde automatisch durch kapazitätszustand fixiert.
  wb_kapa_mo            NUMERIC(8,2),                   --Maschinenkapazität
  wb_tkapa_mo           NUMERIC(8,2),                   --Turbokapazität
  wb_kapa_di            NUMERIC(8,2),
  wb_tkapa_di           NUMERIC(8,2),
  wb_kapa_mi            NUMERIC(8,2),
  wb_tkapa_mi           NUMERIC(8,2),
  wb_kapa_do            NUMERIC(8,2),
  wb_tkapa_do           NUMERIC(8,2),
  wb_kapa_fr            NUMERIC(8,2),
  wb_tkapa_fr           NUMERIC(8,2),
  wb_kapa_sa            NUMERIC(8,2),
  wb_tkapa_sa           NUMERIC(8,2),
  wb_kapa_so            NUMERIC(8,2),
  wb_tkapa_so           NUMERIC(8,2),
  -- todo weg oder korrekt? Somit wäre auch das Suchen Timeslots schneller?
  wb_belast             NUMERIC(8,2),                   --Belastung Maschinenkapazität
  -- weg!
  wb_tbelast            NUMERIC(8,2)                    --Belastung Turbokapazität
 );

CREATE INDEX wkst_kapbelast_ks ON wkst_kapbelast(wb_ks);
CREATE INDEX wkst_kapbelast_week ON wkst_kapbelast(wb_week);
CREATE INDEX wkst_kapbelast_ks_week ON wkst_kapbelast(wb_ks, wb_week);


CREATE OR REPLACE FUNCTION wkst_kapbelast__b_i() RETURNS TRIGGER AS $$
    BEGIN
     new.wb_comment:=wb_comment FROM wkst_kapbelast WHERE wb_ks=new.wb_ks AND wb_week<new.wb_week ORDER BY wb_week DESC LIMIT 1;
     RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER wkst_kapbelast__b_i
    BEFORE INSERT
    ON wkst_kapbelast
    FOR EACH ROW
    EXECUTE PROCEDURE wkst_kapbelast__b_i();


CREATE OR REPLACE FUNCTION wkst_kapbelast__b_u() RETURNS TRIGGER AS $$
    BEGIN
     --Auslastung aus getksview setzen
     IF COALESCE(new.wb_belast,0)<>COALESCE(old.wb_belast,0) THEN
            RETURN new;
     END IF;
     --
     IF new.wb_kapa_mo IS NOT NULL OR new.wb_kapa_di IS NOT NULL OR new.wb_kapa_mi IS NOT NULL OR new.wb_kapa_do IS NOT NULL OR new.wb_kapa_fr IS NOT NULL THEN
             IF new.wb_kapa_mo IS NULL THEN
                new.wb_kapa_mo:=ks_ba*ks_ka2*ks_kf/100 FROM ksv WHERE ks_abt=new.wb_oks;
             END IF;
             IF new.wb_kapa_di IS NULL THEN
                new.wb_kapa_di:=ks_ba*ks_ka3*ks_kf/100 FROM ksv WHERE ks_abt=new.wb_oks;
             END IF;
             IF new.wb_kapa_mi IS NULL THEN
                new.wb_kapa_mi:=ks_ba*ks_ka4*ks_kf/100 FROM ksv WHERE ks_abt=new.wb_oks;
             END IF;
             IF new.wb_kapa_do IS NULL THEN
                new.wb_kapa_do:=ks_ba*ks_ka5*ks_kf/100 FROM ksv WHERE ks_abt=new.wb_oks;
             END IF;
             IF new.wb_kapa_fr IS NULL THEN
                new.wb_kapa_fr:=ks_ba*ks_ka6*ks_kf/100 FROM ksv WHERE ks_abt=new.wb_oks;
             END IF;
     END IF;
     --
     IF   (new.wb_kapa_mo   IS NOT NULL
        OR new.wb_kapa_di   IS NOT NULL
        OR new.wb_kapa_mi   IS NOT NULL
        OR new.wb_kapa_do   IS NOT NULL
        OR new.wb_kapa_fr   IS NOT NULL) THEN

        IF (new.wb_kapa_mo+new.wb_kapa_di+new.wb_kapa_mi+new.wb_kapa_do+new.wb_kapa_fr) IS NULL  THEN
            RAISE EXCEPTION '%', Format(lang_text(29178) /*'Kapazität für alle Wochentage angeben xtt2399'*/);
        END IF;
        --für die Wochentage is ne Kapa angegeben, das Wochenende definieren wir mal mit 0 vor
        new.wb_kapa_sa:=COALESCE(new.wb_kapa_sa,0);
        new.wb_kapa_so:=COALESCE(new.wb_kapa_so,0);
     ELSE
        If COALESCE(new.wb_kapa_sa,0)=0 THEN
            new.wb_kapa_sa:=NULL;
        END IF;
        If COALESCE(new.wb_kapa_so,0)=0 THEN
            new.wb_kapa_so:=NULL;
        END IF;
     END IF;
     --
     RETURN new;
    END $$ LANGUAGE plpgsql;


    CREATE TRIGGER wkst_kap_belast__b_u
    BEFORE UPDATE
    ON wkst_kapbelast
    FOR EACH ROW
    EXECUTE PROCEDURE wkst_kapbelast__b_u();

-- TODO ENTWURF
CREATE TABLE tabk.resource_day_kapa
 (rdk_id                serial PRIMARY KEY,
  rdk_wb_id             integer NOT NULL REFERENCES wkst_kapbelast,
  rdk_day_num           integer,
  rdk_day_time_start    time(0),
  rdk_day_time_end      time(0)
 );

CREATE TABLE ab2_wkstplan_response
 (awpr_id               SERIAL PRIMARY KEY,
  awpr_a2_id            INTEGER NOT NULL REFERENCES ab2 ON DELETE CASCADE,
  awpr_minr             VARCHAR(50),--varchar wegen anderen logins (Fertigung)
  awpr_quitt            BOOL NOT NULL DEFAULT FALSE,
  awpr_end              BOOL NOT NULL DEFAULT FALSE,
  awpr_text             TEXT
 );

CREATE INDEX ab2_wkstplan_response_a2_id ON ab2_wkstplan_response(awpr_a2_id);

/*Rückmeldungen*/

CREATE TABLE rm (
  r_id                  SERIAL PRIMARY KEY,
  r_a2_id               INTEGER NOT NULL REFERENCES ab2 ON DELETE CASCADE,
  r_zeinh               INTEGER NOT NULL REFERENCES zeinh,
  r_da                  TIMESTAMP WITHOUT TIME ZONE DEFAULT currenttime(),
  r_ks                  VARCHAR(9) NOT NULL REFERENCES ksv ON UPDATE CASCADE,
  r_ruest               BOOLEAN NOT NULL DEFAULT FALSE,
  r_std                 NUMERIC,
  r_std_sek             NUMERIC,
  r_minr                INTEGER /*NOT NULL */CONSTRAINT xtt1484__r_minr REFERENCES llv ON UPDATE CASCADE,
  r_mat                 NUMERIC,
  r_as                  NUMERIC,                                                     -- Ausschussmenge
  r_asg_id              INTEGER REFERENCES bdea_ausschussgruende ON UPDATE CASCADE,  -- Ausschussgrund
  r_w_wen               INTEGER REFERENCES wendat ON UPDATE CASCADE ON DELETE CASCADE,          -- Auswärtsrücklieferung, für die diese Rückmeldung steht
  r_belp_id             INTEGER -- REFERENCES belegpos ON UPDATE CASCADE ON DELETE CASCADE -- siehe X TableConstraints.sql -- Rechnung auf Auswärtsrücklieferung, für die diese Rückmeldung steht
 );

-- Indizes
    CREATE INDEX rm_a2_id ON rm(r_a2_id);
    CREATE INDEX rm_r_da ON rm(CAST(r_da AS DATE));
--

--
CREATE OR REPLACE FUNCTION rm__b_iu() RETURNS TRIGGER AS $$
  DECLARE zeinh INTEGER;
  BEGIN
    zeinh:=new.r_zeinh;

    /*1=Stunden*/
    /*2=Minuten*/
    /*3=Sekunden*/

    IF zeinh = 1 THEN
        new.r_std_sek:=new.r_std*3600;
    ELSIF zeinh = 2 THEN
        new.r_std_sek:=new.r_std*60;
    ELSE
        new.r_std_sek:=new.r_std;
    END IF;

    PERFORM disablemodified();
    UPDATE ab2 SET a2_rckmeld=TRUE, a2_buch=FALSE WHERE a2_id=new.r_a2_id AND NOT a2_rckmeld;
    PERFORM enablemodified();
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER rm__b_iu
    BEFORE INSERT OR UPDATE
    ON rm
    FOR EACH ROW
    EXECUTE PROCEDURE rm__b_iu();
--

--
CREATE OR REPLACE FUNCTION rm__a_d() RETURNS TRIGGER AS $$
  DECLARE rck BOOL;
  BEGIN
    rck:=(SELECT TRUE FROM rm WHERE r_a2_id=old.r_a2_id LIMIT 1);
    IF rck IS NULL THEN
        rck:=FALSE;
    END IF;
    --
    PERFORM disablemodified();
    UPDATE ab2 SET a2_rckmeld=rck WHERE a2_id=old.r_a2_id;
    PERFORM enablemodified();
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER rm__a_d
    AFTER DELETE
    ON rm
    FOR EACH ROW
    EXECUTE PROCEDURE rm__a_d();
--

-- Nachkalkulation - Rückmeldungen
  -- gruppiert nach: MA, AG, KS (ggf. KS-AP), tr oder ta - siehe tplanterm.buchrm
  -- Auswärts-AG enthalten keine Zeiten sondern nur Stückzahlen.
  -- Es gibt vier Typen nacherfasster Zeiten:
  --   1. Hauptzeit ( n2_ausw = n2_ruest = n2_pz_para = false )
  --   2. Auswärtsbearbeitung ( n2_ausw = true )
  --   3. Rüstzeit ( n2_ruest = true )
  --   4. Parallele Personalzeit ( n2_pz_para = true )
  -- Letztere wird nicht direkt erfasst, sondern ergibt sich aus Hauptzeit und vorkalkulierten Zeiten,
  --   siehe https://help.prodat-erp.de/index.html?opl_register_op2_ag.html.
  -- Das heißt, dass parallele Personalzeiten nicht separat gestempelt werden dürfen.
CREATE TABLE nk2 (
  n2_id       serial PRIMARY KEY,
  n2_ix       integer REFERENCES abk,                     -- Rückmeldung für ABK
  n2_a2_id    integer REFERENCES ab2 ON DELETE SET NULL,  -- Rückmeldung für Arbeitsgang-ID
  n2_n        integer NOT null,                           -- Rückmeldung für Arbeitsgang-Nr.
  n2_ks       varchar(9),                                 -- Kostenstelle
  n2_ksap     varchar(50),                                -- Kostenstellen-Arbeitsplatz
  n2_ausw     boolean NOT null DEFAULT false,             -- Auswärtsvergabe
  n2_ll       integer REFERENCES llv ON UPDATE CASCADE,   -- Mitarbeiter-Nr.
  n2_ez_stu   numeric(12,4),                              -- effektive Zeit in Stunden
  n2_ruest    boolean NOT null DEFAULT false,             -- Rüst- oder Ausführungszeit
  n2_txt      text,                                       -- Hinweistext (nicht Arbeitsgang-Text)
  n2_mat      numeric,                                    -- zurückgemeldete Stückzahl des MA an AG, KS(-AP) - siehe tplanterm.buchrm
  n2_as       numeric,                                    -- zurückgemeldeter Ausschuss
  n2_pz_para  boolean NOT null DEFAULT false              -- parallele Personalzeit
);

-- Indizes
  -- Arbeitsgang, Kostenstelle, Mitarbeiter, Rüsten, parallele Personalzeit
  CREATE UNIQUE INDEX nk2_unique ON nk2 (n2_a2_id, n2_ks, n2_ll, n2_ruest, n2_pz_para);
  CREATE INDEX nk2_n2_ix ON nk2 (n2_ix);
--

--
CREATE OR REPLACE FUNCTION nk2__a_iu() RETURNS TRIGGER AS $$
  BEGIN
    -- verwendete Kostenstellen mit Stundensätzen aus AVOR für NK übertragen
    IF NOT EXISTS(SELECT true FROM nkksv WHERE nkk_ab_ix = new.n2_ix AND nkk_ks = new.n2_ks) THEN
        INSERT INTO nkksv (
            nkk_ab_ix,  nkk_ks,   -- ABK an Kostenstelle
            -- Norm- und Grenzstundensätze
            nkk_nssf,   nkk_gssf, -- Fertigung (Ausführung)
            nkk_nssr,   nkk_gssr, -- Rüsten
            nkk_nssm,   nkk_gssm  -- Bedienung (Personalzeit)
          )
        SELECT DISTINCT ON ( ks_abt )
            new.n2_ix,  ks_abt,
            -- Norm- und Grenzstundensätze aus KS und überschriebener Stundensatz aus ASK
            coalesce(o2_sts, ks_sts), ks_gss,
            ks_stsr,    ks_gssr,
            ks_stsm,    ks_gssm
        FROM ksv
          LEFT JOIN ab2 ON a2_id = new.n2_a2_id
          LEFT JOIN abk ON ab_ix = a2_ab_ix
          LEFT JOIN opl ON op_ix = ab_askix
          LEFT JOIN op2 ON o2_ix = op_ix AND o2_n = a2_n AND o2_ks = a2_ks
        WHERE ks_abt = new.n2_ks
        -- überschriebenen Stundensatz aus ASK bevorzugen, vgl. UNIQUE INDEX nkksv_index
        ORDER BY ks_abt, o2_sts IS NOT NULL DESC
        ;

    END IF;

    -- Bei Änderung eines AGs in NK zugehörige ABK auf geändert setzen.
    UPDATE abk SET ab_nk_change = true WHERE ab_ix = new.n2_ix AND NOT ab_nk_change;


    -- Verbuchung nicht für Administration kennzeichnen
    IF TSystem.current_user_in_syncro_dblink() THEN
        RETURN new;
    END IF;


    -- Verbuchung in ABK und ABK-AG kennzeichnen
    PERFORM disablemodified();

    UPDATE abk SET ab_buch = true WHERE ab_ix = new.n2_ix    AND NOT ab_buch;
    UPDATE ab2 SET a2_buch = true WHERE a2_id = new.n2_a2_id AND NOT a2_buch;

    PERFORM enablemodified();


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER nk2__a_iu
    AFTER INSERT OR UPDATE
    ON nk2
    FOR EACH ROW
    EXECUTE PROCEDURE nk2__a_iu();

--
CREATE OR REPLACE FUNCTION nk2__a_d() RETURNS TRIGGER AS $$
  BEGIN
    -- Beim Löschen gibt es keine weiteren Rückmeldungen für diesen ABK-AG
    IF NOT EXISTS(SELECT true FROM nk2 WHERE n2_a2_id = old.n2_a2_id AND n2_id <> old.n2_id) THEN
        -- Verbucht-Status in ABK-AG zurücksetzen.
        UPDATE ab2 SET a2_buch = false WHERE a2_id = old.n2_a2_id;
    END IF;


    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER nk2__a_d
    AFTER DELETE
    ON nk2
    FOR EACH ROW
    EXECUTE PROCEDURE nk2__a_d();
--

-- Ein Datensatz für die Nachkalkulation kann nicht mehrere der Eigenschaften Auswärtsbearbeitung, Rüstkosten und parallele Personalzeit haben.
CREATE OR REPLACE FUNCTION nk2__b_iu__flags() RETURNS TRIGGER AS $$
  BEGIN

    IF ( new.n2_ausw::integer + new.n2_ruest::integer + new.n2_pz_para::integer ) >= 2 THEN
      RAISE EXCEPTION 'Entweder Rüstzeit oder Auswärtsbearbeitung oder parallele Personalzeit oder nichts davon. xtt28750';
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER nk2__b_iu__flags
    BEFORE INSERT OR UPDATE OF n2_ausw, n2_ruest, n2_pz_para
    ON nk2
    FOR EACH ROW
    EXECUTE PROCEDURE nk2__b_iu__flags();
--


-- Stundensätze der verwendeten Kostenstellen für Nachkalkulation (NK) und Unfertige Erzeugnisse (UE)
-- vgl. Kostenstelle: ksv
CREATE TABLE nkksv(
 nkk_id                 SERIAL PRIMARY KEY,
  nkk_ab_ix       integer NOT NULL REFERENCES abk ON UPDATE CASCADE ON DELETE CASCADE,  -- für diese ABK gültig
  nkk_ks          varchar(9),     -- Kostenstelle bei ABK in Verwendung
  nkk_nssf        numeric(10,2),  -- Normstundensatz  Fertigung (Ausführung)
  nkk_nssr        numeric(10,2),  -- Normstundensatz  Rüsten
  nkk_gssf        numeric(10,2),  -- Grenzstundensatz Fertigung (Ausführung)
  nkk_gssr        numeric(10,2),  -- Grenzstundensatz Rüsten
  nkk_nssm        numeric(10,2),  -- Normstundensatz  Bedienung (Personalzeit)
  nkk_gssm        numeric(10,2)   -- Grenzstundensatz Bedienung (Personalzeit)
 );

CREATE UNIQUE INDEX nkksv_index ON nkksv (nkk_ab_ix, nkk_ks);


-- globale Zuschläge für Nachkalkulation (NK) und Unfertige Erzeugnisse (UE), prozentual
-- vgl. ASK: op7zko und kostrechgemko
CREATE TABLE nk7zko(
  n7zk_id       serial PRIMARY KEY,
  n7zk_ab_ix    integer NOT NULL REFERENCES abk ON UPDATE CASCADE ON DELETE CASCADE,  -- für diese ABK gültig
  n7zk_krc_bez  varchar(75),    -- Bezeichnung des Zuschlags
  n7zk_proz     numeric(12,6)   -- Satz des Zuschlags
);

-- ohne UNIQUE INDEX, vgl. Quelle
CREATE INDEX nk7zko__abk_krc ON nk7zko( n7zk_ab_ix, n7zk_krc_bez );


/*Auswaertsbearbeitung*/

--Kategorien Lieferanten
CREATE TABLE auswkategorie
(awk_bez                VARCHAR(50) PRIMARY KEY,
 awk_lief               BOOL NOT NULL DEFAULT false,
 awk_auswb              BOOL NOT NULL DEFAULT false,
 awk_txt                TEXT                    --Standard-Arbeitsgangtext, wird in ASK-AG übernommen bei Kategorieauswahl
);


CREATE OR REPLACE FUNCTION ausw_get_menge_ausw(a2id INTEGER) RETURNS NUMERIC(12,4) AS $$ -- TODO weg
    BEGIN
     RETURN tplanterm.ausw_menge_send(a2id);
    END $$ LANGUAGE plpgsql STABLE;


CREATE TABLE nkz
 (nz_id                 SERIAL PRIMARY KEY,
  nz_ab_ix              INTEGER NOT NULL REFERENCES abk ON DELETE CASCADE,
  nz_pos                INTEGER,
  nz_text               VARCHAR(75),
  nz_ad_krz             VARCHAR(21) REFERENCES adk ON UPDATE CASCADE,
  nz_renr               VARCHAR(50),
  nz_redat              DATE DEFAULT current_date,
  nz_stk                NUMERIC,
  nz_me                 VARCHAR(10),
  nz_ep                 NUMERIC(12,2),
  nz_op3_ep             NUMERIC(12,2),
  nz_betrag             NUMERIC(12,2),
  nz_notInCalc          BOOL DEFAULT FALSE NOT NULL  -- Wenn TRUE werden Kosten nicht in Einzelkosten der Nachkalkulation eingerechnet.
 );


/*PLANTAFEL TABELLEN*/

CREATE TABLE plant_register
 (--ptr_id              SERIAL,
  ptr_name              varchar PRIMARY KEY,
  ptr_num               integer,
  ptr_visible           boolean NOT NULL DEFAULT true -- Derzeit nur intern zur Erstellung von DemoSystem
 );

CREATE TABLE plant_ks
 (pt_id                 SERIAL PRIMARY KEY,
  pt_ks                 VARCHAR(9) REFERENCES ksv ON UPDATE CASCADE,
  pt_ptr_name           VARCHAR REFERENCES plant_register ON UPDATE CASCADE,
  pt_top                INTEGER,
  pt_left               INTEGER,
  pt_width              INTEGER,
  pt_height             INTEGER
 );


CREATE OR REPLACE FUNCTION ausschusskosten(abix INTEGER) RETURNS NUMERIC AS $$
  DECLARE matkost         NUMERIC;
          kskost          NUMERIC;
          asanteil        NUMERIC; --Verhältnis Ausschuß/Hergestellt
          sumKost         NUMERIC;
          rec             RECORD;
          abkstk          NUMERIC;
          _pz_para_kost   numeric;
  BEGIN
    sumKost:= 0;
    --Alle ABG der ABK holen, wo Ausschuss anfiel
    FOR rec IN (SELECT * FROM nk2 JOIN abk ON n2_ix=ab_ix WHERE (abix = n2_ix) AND (n2_as > 0)) LOOP

        matkost := 0;
        kskost := 0;

        --Ausschuß im Arbeitsgang / (Menge gefertigt+Menge Ausschuss)
        asanteil:=(COALESCE(rec.n2_as,0) / ( rec.ab_nk_st_uf1 + rec.ab_nk_auss));

        --Anteilig Materialkosten. Keine ABG-Bindung des Materials => Anteil am Gesamtmaterial der ABK.
        SELECT COALESCE(SUM(COALESCE(ag_nk_vkp_uf1, ag_nk_calc_vkp_uf1, ag_vkp_uf1) * ag_stkl),0) INTO matkost FROM auftg WHERE ag_parentabk = abix;
        matkost:= matkost*asanteil;

        --Summe über (MaschinenKosten*Laufzeit) bis zu dem Arbeitsgang wo es Ausschuß gab
        kskost := ( SELECT kosten_ausfuehrung FROM tplanterm.nk__kosten_fertigung_tr_ta__sum( abix, rec.n2_n ) );

        --Summe der parallelen Personalzeit bis zu dem Arbeitsgang wo es Ausschuß gab
        _pz_para_kost := tplanterm.nk__pz_para__sum( abix, rec.n2_n );


        kskost:=kskost*asanteil;
        --RAISE NOTICE 'ABG: % - MatKost: % - KS-Kost: % - Anteil: %', rec.n2_n, ROUND(matkost,2) , ROUND(kskost,2), asanteil;
        sumKost := sumkost + matkost + kskost + _pz_para_kost;

    END LOOP;

    RETURN COALESCE(sumKost,0);
  END $$ LANGUAGE plpgsql STABLE;
--

-- Arbeitsgänge inlusive Rückgaben kopieren
CREATE OR REPLACE FUNCTION copyAG_toABK(
        IN inTrg_ABK               integer,
        IN inSrc_ABK               integer,
        IN inSrc_AB2dbrid          varchar   DEFAULT NULL,
        IN withResultValues        boolean   DEFAULT FALSE,
        IN inAgNrStart             integer   DEFAULT NULL,
        IN inAgNrEnd               integer   DEFAULT NULL,
        IN inAT                    timestamp DEFAULT NULL,
        IN inET                    timestamp DEFAULT NULL,
        IN cyc_kopie_ab2ba         boolean   DEFAULT FALSE,
        IN cyc_kopie_oplpm_data    boolean   DEFAULT FALSE
    ) RETURNS VOID AS $$
  DECLARE ab2rec RECORD;
          newab2id INTEGER;
  BEGIN

    FOR ab2rec IN SELECT * FROM ab2 WHERE (a2_ab_ix = inSrc_ABK OR ab2.dbrid = inSrc_AB2dbrid) AND ((a2_n BETWEEN inAgNrStart AND inAgNrEnd) OR inAgNrStart IS NULL OR inAgNrEnd IS NULL) ORDER BY a2_n LOOP
        INSERT INTO ab2 (a2_ab_ix, a2_dlz, a2_at, a2_et, a2_ks, a2_ksap, a2_v_ll_dbusename,
            a2_ncnr, /*a2_ll_minr, */a2_prio, a2_aknr, a2_ausw, a2_awpreis, a2_adkrz, a2_awtx, a2_tv,
            a2_zeinh_tr, a2_zeinh_tx, a2_tr, a2_tr_sek, a2_th, a2_th_sek, a2_tm, a2_tm_sek, a2_tn, a2_tn_sek,
            a2_lgz, a2_lgz_sek, a2_ul, a2_subject, a2_txt, a2_txt_rtf, a2_txt_intern, a2_interm, a2_msp_id)
        SELECT inTrg_ABK, ab2rec.a2_dlz, inAT, inET, ab2rec.a2_ks, ab2rec.a2_ksap,
            ab2rec.a2_v_ll_dbusename, ab2rec.a2_ncnr, /*ab2rec.a2_ll_minr, */ab2rec.a2_prio, ab2rec.a2_aknr,
            ab2rec.a2_ausw, ab2rec.a2_awpreis, ab2rec.a2_adkrz, ab2rec.a2_awtx, ab2rec.a2_tv, ab2rec.a2_zeinh_tr,
            ab2rec.a2_zeinh_tx, ab2rec.a2_tr, ab2rec.a2_tr_sek, ab2rec.a2_th, ab2rec.a2_th_sek, ab2rec.a2_tm,
            ab2rec.a2_tm_sek, ab2rec.a2_tn, ab2rec.a2_tn_sek, ab2rec.a2_lgz, ab2rec.a2_lgz_sek, ab2rec.a2_ul,
            ab2rec.a2_subject, ab2rec.a2_txt, ab2rec.a2_txt_rtf, ab2rec.a2_txt_intern, ab2rec.a2_interm, ab2rec.a2_msp_id
        RETURNING a2_id INTO newab2id
        ;

        -- Alle Rückgabeparameter mitgeben, neue ID per RETURNING
        INSERT INTO ab2_resultparam (a2r_a2_id, a2r_vname, a2r_vbez, a2r_type, a2r_force, a2r_txt, a2r_value, a2r_bem)
        SELECT newab2id, a2r_scr.a2r_vname, a2r_scr.a2r_vbez, a2r_scr.a2r_type, a2r_scr.a2r_force, a2r_scr.a2r_txt,
            IFTHEN(withResultValues, a2r_scr.a2r_value, NULL), IFTHEN(withResultValues, a2r_scr.a2r_bem, NULL)
        FROM ab2_resultparam AS a2r_scr
        WHERE a2r_scr.a2r_a2_id = ab2rec.a2_id;
        ---

        IF cyc_kopie_ab2ba THEN
             INSERT INTO ab2ba( a2ba_ak_nr, a2ba_noz_id, a2ba_a2_id, a2ba_ab_ix, a2ba_txt )
             SELECT a2ba_ak_nr, a2ba_noz_id, newab2id  , a2ba_ab_ix, a2ba_txt
             FROM ab2ba
             WHERE a2ba_a2_id = ab2rec.a2_id;
        END IF;

        IF cyc_kopie_oplpm_data THEN
             INSERT INTO oplpm_data( pm_op2_id, pm_a2_id , pm_pmnr, pm_pnkt, pm_part, pm_nenn, pm_tol1, pm_tol2, pm_tol3, pm_ma,
                                      pm_pi, pm_fd, pm_sta, pm_txt, pm_intv_typ, pm_messwert_bool, pm_einrichten, pm_formel, pm_hinweis )
             SELECT pm_op2_id, newab2id, pm_pmnr, pm_pnkt, pm_part, pm_nenn, pm_tol1, pm_tol2, pm_tol3, pm_ma,
                    pm_pi, pm_fd, pm_sta, pm_txt, pm_intv_typ, pm_messwert_bool, pm_einrichten, pm_formel, pm_hinweis
             FROM oplpm_data
             WHERE pm_a2_id = ab2rec.a2_id;
        END IF;

    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Materialliste kopieren
CREATE OR REPLACE FUNCTION copyMAT_toABK(IN inTrg_ABK INTEGER, IN inSrc_ABK INTEGER) RETURNS VOID AS $$
  DECLARE TrgAG_nr VARCHAR;
          TrgAN_nr VARCHAR;
  BEGIN
      -- ag_nr wird bei Projekt, QAB oder Vertrag über auftg__b_i erstellt.
      -- Bei interner Fertigung anhand Bestellnummer. Muss hier vergeben werden, da nicht per Trigger. Vgl. Verwendung tplanterm.make_abkrmat in Oberfläche.
      TrgAG_nr:=ld_auftg FROM abk JOIN ldsdok ON ab_ld_id=ld_id WHERE ab_ix = inTrg_ABK;
      TrgAN_nr:=ab_an_nr FROM abk WHERE ab_ix = inTrg_ABK;
      -- Rest der auftg-Inhalte: siehe auftg-Trigger
      INSERT INTO auftg (ag_astat,    ag_nr, ag_parentabk, ag_aknr, ag_o6_dimi, ag_o6_stkz, ag_o6_lz, ag_o6_bz, ag_o6_hz, ag_o6_zz, ag_o6_zme, ag_mcv, ag_stk, ag_vkp, ag_nk_vkp, ag_nident, ag_an_nr)
                       SELECT 'I', TrgAG_nr,    inTrg_ABK, ag_aknr, ag_o6_dimi, ag_o6_stkz, ag_o6_lz, ag_o6_bz, ag_o6_hz, ag_o6_zz, ag_o6_zme, ag_mcv, ag_stk, ag_vkp, ag_nk_vkp, ag_nident, TrgAN_nr
      FROM auftg WHERE ag_parentabk = inSrc_ABK AND ag_astat = 'I';
  END $$ LANGUAGE plpgsql;
--

-- komplette ABK-Struktur rekursiv kopieren
-- mit AGs und Rückgabeparametern, optional mit Materiallisten
-- benötigt: Ziel-dbrid, Ziel-Tabellenname, Ziel-Schlüsselwert(table primary key), Quell-ABK-Index
-- optional: mit Material
-- Bsp. siehe RTF 16225
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
CREATE OR REPLACE FUNCTION copyABK_toDbrid(
        IN inTrg_Dbrid             varchar,
        IN inTrgAb_tablename       varchar,
        IN inTrgAb_keyvalue        varchar,
        IN INSRCAB_IX              integer,
        IN withMat                 boolean DEFAULT FALSE,
        IN withResultValues        boolean DEFAULT FALSE,
        IN inMainAb_ix             integer DEFAULT NULL,
        IN inParentAb_ix           integer DEFAULT NULL,
        IN incyc_kopie_ab2ba       boolean DEFAULT FALSE,
        IN incyc_kopie_oplpm_data  boolean DEFAULT FALSE
    ) RETURNS VOID AS $$
  DECLARE newabix integer;
          subabkab_ix integer;
  BEGIN

    -- ? ab_st_uf1, ab_op_stat
    -- ab_askix und ab_ap_nr holen automatisch alles vom AP
    -- über ab_askix = -1 wird das per Trigger abk__b_iu__abapnr abgefangen, dann NULL
    INSERT INTO abk (ab_ap_nr, ab_askix, ab_ap_bem, ab_ap_txt, ab_ap_txt_rtf,
        ab_parentabk, ab_mainabk, ab_tablename, ab_keyvalue, ab_dbrid,
        ab_pos, ab_st_uf1, ab_an_nr)
    SELECT COALESCE(ab_ap_nr,''), -1, ab_ap_bem, ab_ap_txt, ab_ap_txt_rtf,
        inParentAb_ix, inMainAb_ix, inTrgAb_tablename, inTrgAb_keyvalue, inTrg_Dbrid,
        ab_pos, ab_st_uf1, ab_an_nr
    FROM abk AS scr_abk
    WHERE scr_abk.ab_ix = inSrcAb_ix
    RETURNING ab_ix INTO newabix
    ;

    -- Übergabe Hauptabk, durchschleifen
    IF inMainAb_ix IS NULL THEN
        inMainAb_ix:=newabix;
    END IF;

    -- Arbeitsgänge mit Rückgaben ziehen
    PERFORM copyAG_toABK(
                          inTrg_ABK             => newabix,
                          inSrc_ABK             => inSrcAb_ix,
                          cyc_kopie_ab2ba       => incyc_kopie_ab2ba,
                          cyc_kopie_oplpm_data  => incyc_kopie_oplpm_data,
                          inSrc_AB2dbrid        => NULL,
                          withResultValues      => withResultValues
                        );

    -- Materiallisten mitkopieren
    IF withMat THEN
        PERFORM copyMAT_toABK(newabix, inSrcAb_ix);
    END IF;

    -- Rekursion
    FOR subabkab_ix IN SELECT ab_ix FROM abk WHERE ab_parentabk = inSrcAb_ix LOOP
        PERFORM copyABK_toDbrid( inTrg_Dbrid             => inTrg_Dbrid,
                                  inTrgAb_tablename      => inTrgAb_tablename,
                                  inTrgAb_keyvalue       => inTrgAb_keyvalue,
                                  INSRCAB_IX             => subabkab_ix,
                                  withMat                => withMat,
                                  withResultValues       => withResultValues,
                                  inMainAb_ix            => inMainAb_ix,
                                  inParentAb_ix          => newabix,
                                  incyc_kopie_ab2ba      => incyc_kopie_ab2ba,
                                  incyc_kopie_oplpm_data => incyc_kopie_oplpm_data
                                );
    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql;
--


-- Funktion übergibt die Datensätze direkt aus dem VIEW ~ kein Neuladen der betroffenen Daten notwendig; Vorteil VIEW-Feld ist kundengesteuert
CREATE OR REPLACE FUNCTION treporting.NormZert_customer_descr(IN src_normzert normzert, IN src_qsnorm qsnorm, IN src_qsnorm_descr qsnorm_descr, src_pre_NormZertTyp pre_NormZertTyp) RETURNS VARCHAR AS $$
  BEGIN
    IF TSystem.Settings__Get('KUNDE') IN ('LOLL','LOLL-MECHATRONIK') THEN
      RETURN concat_ws(' ', src_qsnorm.qs_ident, src_normzert.noz_allg1); --Norm-Ident+Zusatz1
    ELSE
      RETURN concat_ws(' ', src_qsnorm.qs_ident, src_pre_NormZertTyp.pnzt_bez_sub, src_qsnorm_descr.qsd_bez, src_normzert.noz_idx); --Norm-Ident+Normenart+Kurzbeschreibung+Index
    END IF;
  END $$ LANGUAGE plpgsql STABLE;
--

-- #10009 Projektnachkalkulation
CREATE OR REPLACE FUNCTION anl__projekt_nk_calc (
    IN _ab_ap_nr              VARCHAR,
    OUT ab_mainabk            INTEGER,
    OUT ab_parentabk          INTEGER,
    OUT ab_ix                 INTEGER,
    OUT ab_ap_nr              VARCHAR,
    OUT arbeitszeit_soll      NUMERIC,
    OUT arbeitszeit_ist       NUMERIC,
    OUT arbeitszeit_kost_soll NUMERIC,
    OUT arbeitszeit_kost_ist  NUMERIC,
    OUT mat_kost_soll         NUMERIC,
    OUT mat_kost_ist          NUMERIC,
    OUT roh_mat_kost_soll     NUMERIC,
    OUT roh_mat_kost_ist      NUMERIC,
    OUT norm_mat_kost_soll    NUMERIC,
    OUT norm_mat_kost_ist     NUMERIC,
    OUT aw_kost_soll          NUMERIC,
    OUT aw_kost_ist           NUMERIC,
    OUT abk_kost_soll         NUMERIC,
    OUT abk_kost_ist          NUMERIC,
    OUT sond_kost_ist         NUMERIC,
    OUT abk_stk_kost_soll     NUMERIC,
    OUT abk_stk_kost_ist      NUMERIC
  ) RETURNS SETOF RECORD AS $$
  DECLARE rec RECORD;
  BEGIN
    --Erledigte Positionen sind schon gebucht oder sollen übersprungen werden.
    FOR rec IN ( SELECT abk.ab_mainabk, abk.ab_parentabk, abk.ab_ix, abk.ab_ap_nr FROM abk WHERE (ab_tablename='anl' AND ab_dbrid = (SELECT dbrid FROM anl WHERE an_nr = _ab_ap_nr LIMIT 1)) ORDER BY ab_ix ) LOOP
        -- ABK-Preise aktualisieren
        PERFORM tplanterm.auftg_nk_calc_vkp(rec.ab_ix);

        ab_mainabk:=    rec.ab_mainabk;
        ab_parentabk:=  rec.ab_parentabk;
        ab_ix:=         rec.ab_ix;
        ab_ap_nr:=      rec.ab_ap_nr;

        SELECT ak.arbeitszeit_soll,   ak.arbeitszeit_ist,   ak.arbeitszeit_kost_soll, ak.arbeitszeit_kost_ist,                                                -- Arbeitszeit
               ak.mat_kost_soll,      ak.mat_kost_ist,      ak.roh_mat_kost_soll,     ak.roh_mat_kost_ist,      ak.norm_mat_kost_soll,  ak.norm_mat_kost_ist, -- Material
               ak.aw_kost_soll,       ak.aw_kost_ist,                                                                                                         -- Auswärts
               ak.sond_kost_ist,                                                                                                                              -- Sonderkosten
               ak.abk_kost_soll,      ak.abk_kost_ist,      ak.abk_stk_kost_soll,     ak.abk_stk_kost_ist                                                     -- gesamt

        INTO      arbeitszeit_soll,      arbeitszeit_ist,      arbeitszeit_kost_soll,    arbeitszeit_kost_ist,
                  mat_kost_soll,         mat_kost_ist,         roh_mat_kost_soll,        roh_mat_kost_ist,         norm_mat_kost_soll,     norm_mat_kost_ist,
                  aw_kost_soll,          aw_kost_ist,
                  sond_kost_ist,
                  abk_kost_soll,         abk_kost_ist,          abk_stk_kost_soll,       abk_stk_kost_ist
        FROM tabk.get_abk_kosten(rec.ab_ix, 2) AS ak;

        RETURN NEXT;
    END LOOP;
  END $$ LANGUAGE plpgsql;
--

-- #11766 [Splitt-ABK] Kostenübersicht
CREATE OR REPLACE FUNCTION TAbk.abk__split__nk_calc (
    IN _ab_ix                 INTEGER,
    OUT ab_mainabk            INTEGER,
    OUT ab_parentabk          INTEGER,
    OUT ab_ix                 INTEGER,
    OUT ab_ap_nr              VARCHAR,
    OUT arbeitszeit_soll      NUMERIC,
    OUT arbeitszeit_ist       NUMERIC,
    OUT arbeitszeit_kost_soll NUMERIC,
    OUT arbeitszeit_kost_ist  NUMERIC,
    OUT mat_kost_soll         NUMERIC,
    OUT mat_kost_ist          NUMERIC,
    OUT roh_mat_kost_soll     NUMERIC,
    OUT roh_mat_kost_ist      NUMERIC,
    OUT norm_mat_kost_soll    NUMERIC,
    OUT norm_mat_kost_ist     NUMERIC,
    OUT aw_kost_soll          NUMERIC,
    OUT aw_kost_ist           NUMERIC,
    OUT abk_kost_soll         NUMERIC,
    OUT abk_kost_ist          NUMERIC,
    OUT sond_kost_ist         NUMERIC,
    OUT abk_stk_kost_soll     NUMERIC,
    OUT abk_stk_kost_ist      NUMERIC
  ) RETURNS SETOF RECORD AS $$
  DECLARE rec RECORD;
  BEGIN
    --Erledigte Positionen sind schon gebucht oder sollen übersprungen werden.
    FOR rec IN ( SELECT abk.ab_ix, abk.ab_mainabk, abk.ab_parentabk, abk.ab_stat, abk.ab_ap_nr
                 FROM abk
                 WHERE (abk.ab_parentabk = _ab_ix AND TSystem.ENUM_getValue(ab_stat,  'SPL')) OR abk.ab_ix = _ab_ix
                 ORDER BY ab_ix  ) LOOP
        -- ABK-Preise aktualisieren
        PERFORM tplanterm.auftg_nk_calc_vkp(rec.ab_ix);

        ab_mainabk:=    rec.ab_mainabk;
        ab_parentabk:=  rec.ab_parentabk;
        ab_ix:=         rec.ab_ix;
        ab_ap_nr:=      rec.ab_ap_nr;

        SELECT ak.arbeitszeit_soll,   ak.arbeitszeit_ist,   ak.arbeitszeit_kost_soll, ak.arbeitszeit_kost_ist,                                                -- Arbeitszeit
               ak.mat_kost_soll,      ak.mat_kost_ist,      ak.roh_mat_kost_soll,     ak.roh_mat_kost_ist,      ak.norm_mat_kost_soll,  ak.norm_mat_kost_ist, -- Material
               ak.aw_kost_soll,       ak.aw_kost_ist,                                                                                                         -- Auswärts
               ak.sond_kost_ist,                                                                                                                              -- Sonderkosten
               ak.abk_kost_soll,      ak.abk_kost_ist,      ak.abk_stk_kost_soll,     ak.abk_stk_kost_ist                                                     -- gesamt

        INTO      arbeitszeit_soll,      arbeitszeit_ist,      arbeitszeit_kost_soll,    arbeitszeit_kost_ist,
                  mat_kost_soll,         mat_kost_ist,         roh_mat_kost_soll,        roh_mat_kost_ist,         norm_mat_kost_soll,     norm_mat_kost_ist,
                  aw_kost_soll,          aw_kost_ist,
                  sond_kost_ist,
                  abk_kost_soll,         abk_kost_ist,          abk_stk_kost_soll,       abk_stk_kost_ist
        FROM tabk.get_abk_kosten(rec.ab_ix, 2) AS ak;

        RETURN NEXT;
    END LOOP;
  END $$ LANGUAGE plpgsql;
--

-- Erstellung von Arbeitsgangbeschreibung #10429
CREATE OR REPLACE FUNCTION tabk.abk__ab2__infotext(
    abix INTEGER,
    agnr INTEGER
  ) RETURNS VARCHAR AS $$
  DECLARE rec RECORD;
  BEGIN
    SELECT
      COALESCE(ld_akbz, ak_bez)                                             AS Bezeichnung,
      op_n || ' ' || IFTHEN(ak_allgb2, lang_text(10517), '')                AS Artikel,
      IFTHEN(q_nr IS NOT NULL, 'Nachbearbeitung QAB-Nr.: ' || q_nr, ad_fa1) AS Besteller,
      IFTHEN(COALESCE(trim(o2_rustinfo), '') <> '', lang_text(3828) || ': ' || o2_rustinfo || E'\r\n', '') ||
          IFTHEN(COALESCE(trim(a2_ncnr), '') = '' OR NOT ks_mitnc, '', 'NC-PROGRAMM: ' || a2_ncnr || E'\r\n\r\n') ||
          COALESCE(a2_txt, a2_txt_rtf, '')
      AS Arbeitsanweisung
    FROM abk
      LEFT JOIN LATERAL (SELECT abk2.ab_ld_id AS parent_ab_ld_id FROM abk AS abk2 WHERE abk2.ab_ix = abk.ab_parentabk) AS parent_abk ON abk.ab_ld_id IS NULL AND abk.ab_parentabk IS NOT NULL
      LEFT JOIN ldsdok ON ld_id = COALESCE(ab_ld_id, parent_ab_ld_id)
      LEFT JOIN auftg ON ag_id = tplanterm.abk_main_auftg_id(ld_abk)
      LEFT JOIN opl ON op_ix = ab_askix
      LEFT JOIN qab ON qab.dbrid = ab_dbrid AND ab_tablename = 'qab'
      LEFT JOIN ab2 ON a2_ab_ix = ab_ix
      LEFT JOIN adk ON ad_krz = ag_lkn
      LEFT JOIN ksv ON a2_ks = ks_abt
      LEFT JOIN op2 ON ab2.a2_o2_id = op2.o2_id
      LEFT JOIN art ON COALESCE(op_n, ld_aknr, q_ak_nr) = ak_nr
    WHERE ab_ix = abix
      AND a2_n = agnr
    INTO rec;

    IF rec IS NULL THEN RETURN NULL; END IF;

    RETURN CONCAT(
        -- lang_text(316),   ': ', trim(rec.Bezeichnung), E'\r\n',
        -- lang_text(234),   ': ', trim(rec.Artikel),     E'\r\n',
        lang_text(293),   ': ', trim(rec.Besteller),   E'\r\n',
        lang_text(21038), ': ', trim(rec.Arbeitsanweisung)
    );
  END $$ LANGUAGE plpgsql STABLE;
--

--Funktion für Ermittlung von Standard-/Alternativkostenstellen
CREATE OR REPLACE FUNCTION tabk.ab2__kst_from_ask__get(
  IN  _a2_id               integer,
  IN  _incl_alternativ_kst boolean DEFAULT false,
  OUT ks                   varchar(9),
  OUT ncp_ncnr             varchar(20),
  OUT is_alternative_kst   boolean
  )
  RETURNS SETOF RECORD
  AS $$

  SELECT o2ks_ks AS ks,
         ncp_ncnr,
         true AS is_alternative_kst
    FROM ab2
    JOIN op2 ON a2_o2_id = o2_id
    JOIN op2ksa ON a2_o2_id = o2ks_o2_id                -- Alternativkostenstellen gemäß AVOR zu dem Arbeitsgang
    LEFT JOIN ncp_o2_a2_ksv ON ncpoak_ks    = o2ks_ks   -- NC-Programm zu Alternativkostenstellen finden
                           AND ncpoak_o2_id = o2ks_o2_id
    LEFT JOIN ncprogram ON ncp_id = ncpoak_ncp_id
   WHERE a2_id = _a2_id
     AND _incl_alternativ_kst

  UNION

  -- zulässige Standardkostenstelle aus ASK
  SELECT o2_ks AS ks,
         o2_nc AS ncp_ncnr,
         false AS is_alternative_kst
    FROM ab2
    JOIN op2 ON a2_o2_id = o2_id
   WHERE a2_id = _a2_id;

  $$ LANGUAGE SQL STABLE;


--
CREATE OR REPLACE FUNCTION tabk.abk__filterby(_abk abk, _filtername varchar, _filtervalue varchar)
  RETURNS boolean
  AS $$
  BEGIN
    -- needs rewrite, subquery not needed

    -- valid _filtervalue
    IF ( not( coalesce( _filtervalue, '%' ) = '%' ) AND not( coalesce( _filtervalue, '' ) = '' ) ) THEN
      -- shortcut for ac, because ab2 is not needed
      IF ( LOWER(_filtername) = 'arbeitspaketgruppe' ) THEN
        RETURN
          coalesce
          (
            (
              SELECT
                true
              FROM
                abk
              LEFT JOIN
                art ON ak_nr = ab_ap_nr
              WHERE
                ab_ix = _abk.ab_ix
                AND ( ak_ac = _filtervalue )
            ),
            false
          )
        ;
      ELSE
        -- filters regarding ab2
        RETURN
          EXISTS
          (
            SELECT
              true
            FROM
              ab2
            JOIN
              abk ON ab_ix = a2_ab_ix
            WHERE
              abk.ab_ix = _abk.ab_ix
              AND NOT a2_ende
              AND
                (
                  CASE LOWER(_filtername)
                    WHEN 'arbeitsplatz' THEN
                        a2_ksap::varchar = _filtervalue
                    WHEN 'kostenstelle' THEN
                        a2_ks::varchar = _filtervalue
                    WHEN 'verantwortung' THEN
                        a2_v_ll_dbusename::varchar = _filtervalue
                    WHEN 'benutzer' THEN
                      (
                        ( a2_ks = ( SELECT ll_abteilung FROM llv WHERE ll_db_usename = _filtervalue ) ) -- hier später eine Funktion der Planung verwenden
                        OR ( _filtervalue IN ( SELECT unnest( tplanterm.ksv__ksaps__fetch(a2_ks) ) ) )
                        OR ( a2_v_ll_dbusename = _filtervalue )
                      )
                  ELSE
                    true
                  END
                )
          )
        ;
      END IF;
    ELSE
      -- invalid _filtervalue
      RETURN true;
    END IF;
  END
  $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION tabk.abk__filterby(_abk integer, _filtername varchar, _filtervalue varchar)
  RETURNS boolean
  AS $$
    SELECT tabk.abk__filterby(abk, _filtername, _filtervalue) FROM abk WHERE ab_ix = _abk;
  $$ LANGUAGE sql STABLE;
--